(ns burningswell.api.publisher
  (:refer-clojure :exclude [test])
  (:require [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.events :as events]
            [burningswell.edn :as edn]
            [clojure.walk :as walk]
            [com.stuartsierra.component :as component]
            [jackdaw.client :as client]
            [jackdaw.serdes.edn :as jse]
            [taoensso.timbre :as log])
  (:import org.apache.kafka.clients.producer.Producer))

(def commands-topic
  "The Kafka commands topic."
  {:topic-name "burningswell.api.commands"})

(def events-topic
  "The Kafka events topic."
  {:topic-name "burningswell.api.events"})

(defn- data
  "Convert records in `m` to maps."
  [m]
  (let [f (fn [[k v]]
            [(if (map? k)
               (into {} k) k)
             (if (map? v)
               (into {} v) v)])]
    (walk/postwalk (fn [x]
                     (if (map? x)
                       (into {} (map f x)) x)) m)))

(defrecord Kafka [client config]
  component/Lifecycle
  (start [publisher]
    (let [client (client/producer
                  {"bootstrap.servers" (:bootstrap.servers (:config publisher))}
                  {:key-serde (jse/serde edn/read-opts)
                   :value-serde (jse/serde edn/read-opts)})]
      (log/info {:msg "Kafka publisher successfully started."})
      (assoc publisher :client client)))
  (stop [publisher]
    (some-> publisher :client .close)
    (log/info {:msg "Kafka publisher successfully stopped."})
    (assoc publisher :client nil)))

;; Kafka publisher

(extend-type Kafka
  commands/Publisher
  (-publish [publisher command]
    @(client/produce! (:client publisher) commands-topic (:id command) (data command))
    (log/info {:msg "Sucessfully published command." :command (data command)}))

  events/Publisher
  (-publish [publisher event]
    @(client/produce! (:client publisher) events-topic (:id event) (data event))
    (log/info {:msg "Sucessfully published event." :event (data event)})))

(defn kafka
  "Returns a Kafka publisher component."
  [config]
  (map->Kafka {:config config}))

;; Test publisher

(defn- marshal [x]
  (-> x data edn/prn-str edn/read-string))

(defrecord Test [topics])

(extend-type Test
  commands/Publisher
  (-publish [publisher command]
    (swap! (:topics publisher) update :commands (fnil conj []) (marshal command))
    (log/debug {:msg "Sucessfully published command." :command command}))

  events/Publisher
  (-publish [publisher event]
    (swap! (:topics publisher) update :events (fnil conj []) (marshal event))
    (log/debug {:msg "Sucessfully published event." :event event})))

(defn test
  "Returns a Test publisher component."
  [& [config]]
  (map->Test {:topics (atom {})}))
