(ns burningswell.api.photos
  (:require [burningswell.api.core :as core]
            [burningswell.api.validation :as v]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.db.photos :as photos]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

;; Photos

(s/def :burningswell.api.photos/id
  (s/and ::specs/id (v/db-id-exists? :photos)))

(s/def :burningswell.api.photos/except
  (s/nilable (s/coll-of :burningswell.api.photos/id)))

(s/def :burningswell.api.photos/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.specs/location
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.search/query
                   :burningswell.api.photos/except
                   :burningswell.api.photos/sort]))

(defrecord Photo [id name]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :photo
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} photos]
    (photos/select-batch db photos)))

(defrecord Photos [after before direction distance first except
                   last location sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.photos/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:direction direction
          :distance distance
          :except except
          :limit (core/limit params)
          :location location
          :offset (core/offset params)
          :query query
          :sort sort}
         (photos/search db))))

(defrecord Images [after before direction first label last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.images/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} photos]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :images params)}
         (sql/has-many db photos :photos :images))))

(defrecord Flickr [id]
  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} photos]
    (sql/has-one db photos :public.photos :flickr.photos)))
