(ns burningswell.api.pagination
  (:require [claro.data :as data]
            [burningswell.api.core :as core]
            [claro.data :as data]
            [claro.engine :as engine]
            [claro.runtime.impl :as impl]
            [clojure.edn :as edn]))

(defn encode-cursor [x]
  (core/base64-encode x))

(defn- after-limit [params]
  (or (-> params :after :limit)
      (:first params)
      10))

(defn- after-offset [params]
  (or (-> params :after :offset) 0))

(defn- limit [params]
  (or (:first params) 10))

(defn- offset [params]
  (or (:offset params) 0))

(defn- end-cursor [edges params]
  (let [limit (after-limit params)]
    (encode-cursor
     {:limit limit
      :offset (+ (after-offset params) limit)})))

(defn- start-cursor [edges params]
  (encode-cursor
   {:limit (after-limit params)
    :offset (after-offset params)}))

(defn- has-next-page [edges params]
  (>= (count edges)
      (+ (after-limit params)
         (after-offset params))))

(defn- has-previous-page [edges params]
  (not= 1 (after-offset params)))

;; Cursor

(defrecord Cursor [limit offset]
  data/Cost
  (cost [resolvable batch]
    0)

  data/Resolvable
  (resolve! [params _]
    (into {} params))

  data/Transform
  (transform [_ cursor]
    (core/base64-encode cursor)))

(defn cursor [index params]
  (let [offset (after-offset params)]
    (map->Cursor
     {:limit (after-limit params)
      :offset (+ offset index)})))

;; Page info

(defrecord PageInfo [edges params]
  data/Cost
  (cost [resolvable batch]
    0)

  data/Resolvable
  (resolve! [_ {:keys [db]}]
    {:end-cursor (end-cursor edges params)
     :has-next-page (has-next-page edges params)
     :has-previous-page (has-previous-page edges params)
     :start-cursor (start-cursor edges params)}))

;; Edge

(defrecord Edge [index node resolve-fn params]
  data/Cost
  (cost [resolvable batch]
    0)

  data/Resolvable
  (resolve! [_ {:keys [db]}]
    {:cursor (cursor index params)
     :node (resolve-fn node)}))

(defn- edges [resolveables resolve-fn params]
  (map-indexed #(->Edge %1 %2 resolve-fn params) resolveables))

;; Connection

(defrecord Connection [resolveables resolve-fn params]
  data/Cost
  (cost [resolvable batch]
    0)

  data/Resolvable
  (resolve! [_ {:keys [db]}]
    {:edges (edges resolveables resolve-fn params)
     :page-info (->PageInfo resolveables params)}))
