(ns burningswell.api.page-views
  (:require [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.specs :as specs]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [no.en.core :refer [format-url parse-url]]))

;; Input

(s/def :burningswell.api.page-views.create/input
  (s/keys :req-un [:burningswell.api.specs/url
                   :burningswell.api.specs/session-id]
          :opt-un [:burningswell.api.specs/location]))

(s/def :burningswell.api.page-views/create
  (s/keys :req-un [:burningswell.api.page-views.create/input]))

(defn- safe-url
  "Returns a safe `url`."
  [url]
  (some-> (parse-url (str url))
          (update :query-params dissoc :auth-token)
          (format-url)
          (java.net.URL.)))

(defrecord Create [input]
  commands/Command
  (command [params env]
    (prn (update params :url safe-url))
    {:name :burningswell.api.commands/create-page-view
     :params (update params :url safe-url)})

  conform/Params
  (conform [params env]
    :burningswell.api.page-views/create)

  data/Mutation
  data/Resolvable
  (resolve! [params {:keys [user]}]
    (-> (assoc (:input params) :user-id (:id user))
        (update :url safe-url)))

  events/Events
  (events [resolvable {:keys [user]} result]
    [(cond-> {:name :burningswell.api.events/page-view-created
              :location (:location result)
              :session-id (:session-id result)
              :url (-> result :url safe-url)}
       user
       (assoc :user-id (:id user)))]))
