(ns burningswell.api.oauth.connect
  (:require [burningswell.api.jwt :as jwt]
            [burningswell.api.oauth.providers :as providers]
            [clj-http.client :as http]
            [clj-time.core :as t]
            [inflections.core :as infl]
            [no.en.core :refer [format-url parse-url]]
            [oauth.one :as one]
            [ring.util.response :as response]
            [taoensso.timbre :as log]))

(defn- provider-name
  "Returns the provider name from the :route-params of `request`."
  [request]
  (-> request :route-params :name))

(defn- provider
  "Returns the OAuth provider for `request`."
  [env request]
  (some->> (provider-name request)
           (providers/by-name env)))

(defn- authorization-url
  "Returns the parsed authorization url of `provider`."
  [provider]
  (-> provider :authorization-url parse-url))

(defn- referer
  "Returns the referer from `request`."
  [request]
  (-> (get-in request [:headers "referer"]) parse-url
      (dissoc :query-params)
      format-url))

(defn- state
  "Returns the state for `request`."
  [{:keys [jwt]} request]
  (let [expires (t/plus (t/now) (t/minutes 1))]
    (jwt/sign jwt {:referer (referer request)
                   :remote-addr (:remote-addr request)}
              {:exp expires})))

(defmulti request-token!
  "Fetch the OAuth request token for `provider`."
  (fn [env provider] (-> provider :name keyword)))

(defmethod request-token! :twitter [env provider]
  (let [consumer (providers/oauth-v1-consumer env provider)
        request (one/request-token-request consumer)
        request (assoc request :as :x-www-form-urlencoded)
        {:keys [status body]} (http/request request)]
    (if (= status 200)
      (-> body infl/hyphenate-keys)
      (throw (ex-info "Can't fetch request token." {})))))

(defmethod request-token! :default [env provider])

(defmulti redirect-url
  "Returns the redirect url of the OAuth provider for the `request`."
  (fn [env provider request] (-> provider :name keyword)))

(defmethod redirect-url :facebook [env provider request]
  (merge (authorization-url provider)
         {:query-params
          {:auth_type "rerequest"
           :client_id (-> env :facebook :client-id)
           :redirect_uri (:callback-url provider)
           :response_type "code"
           :scope "public_profile,email"
           :state (state env request)}}))

(defmethod redirect-url :google [env provider request]
  (merge (authorization-url provider)
         {:query-params
          {:access_type "online"
           :client_id (-> env :google :client-id)
           :redirect_uri (:callback-url provider)
           :response_type "code"
           :scope "email profile"
           :state (state env request)}}))

(defmethod redirect-url :linkedin [env provider request]
  (merge (authorization-url provider)
         {:query-params
          {:client_id (-> env :linkedin :client-id)
           :redirect_uri (:callback-url provider)
           :response_type "code"
           :scope "r_basicprofile r_emailaddress"
           :state (state env request)}}))

(defmethod redirect-url :twitter [env provider request]
  (let [token (request-token! env provider)]
    (merge (authorization-url provider)
           {:query-params
            {:oauth_token (:oauth-token token)
             :redirect_uri (:callback-url provider)
             :state (state env request)}})))

(defn- provider-connect [env provider request]
  (let [redirect-url (redirect-url env provider request)]
    (log/info {:msg "Connecting with OAuth provider."
               :provider (:name provider)})
    (-> redirect-url format-url response/redirect)))

(defn- provider-not-found [env request]
  (let [provider (provider-name request)]
    (log/error {:msg "Can't connect with unknown OAuth provider."
                :provider provider})
    {:status 404
     :body (format "OAuth provider \"%s\" not found." provider)}))

(defn connect [env request]
  (if-let [provider (provider env request)]
    (provider-connect env provider request)
    (provider-not-found env request)))
