(ns burningswell.api.middleware.events
  (:require [burningswell.api.middleware.commands :as commands]
            [claro.data.protocols :as claro]
            [claro.engine :as engine]
            [claro.runtime.impl :as impl]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [inflections.core :as infl]
            [potemkin :refer [defprotocol+]]
            [taoensso.timbre :as log]))

(defprotocol+ Events
  (events [resolvable env result]))

(defprotocol+ Publisher
  (-publish [publisher event]))

(extend-type Object
  Events
  (events [resolvable env result]
    nil))

(defn publish!
  "Publish the `event` qqto `publisher`."
  [publisher event]
  (-publish publisher event))

(s/fdef publish!
  :args (s/cat :publisher any? :event :burningswell.api/event))

(defn- make-events
  "Make the events for `resolvable` and `result`."
  [env resolvable result]
  (let [command (-> resolvable meta ::commands/command)
        events (events resolvable env result)]
    (when (and (nil? command) (not (empty? events)))
      (throw (ex-info (str "No command found for resolvable: " (class resolvable))
                      {:events events})))
    (for [event events]
      (cond-> (merge {:id (java.util.UUID/randomUUID)
                      :command-id (:id command)}
                     event)
        (:user env) (assoc :user-id (-> env :user :id))))))

(defn- collect-events
  [env resolvable->result]
  (->> (for [[resolvable result] resolvable->result]
         (if-let [events (seq (make-events env resolvable result))]
           [resolvable (some-> result (assoc ::events events))]
           [resolvable result]))
       (into {})))

(defn- publish-events
  [publisher resolvable->result]
  (->> (for [[resolvable result] resolvable->result]
         (let [events (::events result)]
           (doseq [event events]
             (publish! publisher event)
             (log/debug {:msg "Event successfully published."
                         :event event}))
           [resolvable result]))
       (into {})))

(defn wrap-events
  [engine]
  (let [impl (engine/impl engine)]
    (->> (fn [resolver]
           (fn [env batch]
             (if (every? claro/mutation? batch)
               (impl/chain1 impl (resolver env batch)
                            #(collect-events env %))
               (resolver env batch))))
         (engine/wrap-pre-transform engine))))

(defn wrap-publish-events
  [engine publisher]
  (let [impl (engine/impl engine)]
    (->> (fn [resolver]
           (fn [env batch]
             (if (every? claro/mutation? batch)
               (impl/chain1 impl (resolver env batch)
                            #(publish-events publisher %))
               (resolver env batch))))
         (engine/wrap-pre-transform engine))))
