(ns burningswell.api.middleware.commands
  (:require [claro.data.protocols :as claro]
            [claro.engine :as engine]
            [claro.runtime.impl :as impl]
            [clojure.spec.alpha :as s]
            [potemkin :refer [defprotocol+]]
            [taoensso.timbre :as log]))

(defprotocol+ Command
  (command [resolvable env]))

(defprotocol+ Publisher
  (-publish [publisher command]))

(defn publish!
  "Publish the `command` to `publisher`."
  [publisher command]
  (-publish publisher command))

(s/fdef publish!
  :args (s/cat :publisher any? :command :burningswell.api/command))

(defn- make-command
  "Generates the command for `resolvable`."
  [env resolvable]
  (when-let [command (command resolvable env)]
    (cond-> (merge {:id (java.util.UUID/randomUUID)
                    :params (into {} resolvable)}
                   command)
      (:user env) (assoc :user-id (-> env :user :id)))))

(extend-type Object
  Command
  (command [resolvable env]))

(defn- collect-commands
  [env batch]
  (for [resolvable batch]
    (with-meta resolvable
      {::command (merge (make-command env resolvable)
                        (command resolvable env))})))

(defn- publish-commands
  [publisher resolvable->result]
  (->> (for [[resolvable result] resolvable->result]
         (let [command (-> resolvable meta ::command)]
           (when command
             (publish! publisher command)
             (log/debug {:msg "Command successfully published."
                         :command command}))
           [resolvable result]))
       (into {})))

(defn wrap-commands
  [engine]
  (let [impl (engine/impl engine)]
    (->> (fn [resolver]
           (fn [env batch]
             (if (every? claro/mutation? batch)
               (impl/chain1 impl (collect-commands env batch)
                            #(resolver env %))
               (resolver env batch))))
         (engine/wrap engine))))

(defn wrap-publish-commands
  [engine publisher]
  (let [impl (engine/impl engine)]
    (->> (fn [resolver]
           (fn [env batch]
             (if (every? claro/mutation? batch)
               (impl/chain1 impl (resolver env batch)
                            #(publish-commands publisher %))
               (resolver env batch))))
         (engine/wrap-pre-transform engine))))
