(ns burningswell.api.client
  (:require [burningswell.routes :as routes]
            [cheshire.core :as json]
            [clj-http.client :as http]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [clojure.walk :as walk]
            [grafiko.client :as client]
            [inflections.core :as inf]
            [no.en.core :refer [parse-url]]))

(defprotocol IClient
  (-send [client query variables] "Send the GraphQL `query` to the server."))

(def defaults
  "The default client options."
  {:accept "application/json"
   :as :auto
   :coerce :always
   :content-type "application/json"
   :date-format "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"
   :request-method :post
   :scheme :http
   :server-name "localhost"
   :server-port 7000
   :uri "/graphql"})

;; Client

(defrecord CljHttp [scheme server-name server-port uri]
  IClient
  (-send [client query variables]
    (try (:body (client/query query (assoc client :variables variables)))
         (catch Exception e
           (let [{:keys [body] :as request} (ex-data e)]
             (pprint body)
             (throw (ex-info (with-out-str (pprint body))
                             (or request {})
                             e)))))))

(defn client
  "Returns a new GraphQL client."
  [config]
  (-> (merge defaults config)
      (map->CljHttp)))

(defn q
  "Send the GraphQL `query` to the server."
  [client query & [variables]]
  (-send client query variables))

(defn qp
  "Send the GraphQL `query` to the server and pretty print the result."
  [client query & [variables]]
  (pprint (-send client query variables)))

(defmulti mutation
  (fn [operation] (keyword operation)))

(defmethod mutation :create-spot [_]
  `((mutation
     create-spot
     [($input CreateSpotInput!)]
     (create-spot
      [(input $input)]
      (spot
       id name
       (user id))))))

(defmethod mutation :create-page-view [_]
  `((mutation
     create-page-view
     [($input CreatePageViewInput!)]
     (create-page-view
      [(input $input)]
      session-id url
      (location latitude longitude)
      (user id)))))

(defmethod mutation :dislike-photo [_]
  `((mutation
     dislike-photo
     [($input DislikePhotoInput!)]
     (dislike-photo
      [(input $input)]
      (photo id)
      (user id)))))

(defmethod mutation :like-photo [_]
  `((mutation
     like-photo
     [($input LikePhotoInput!)]
     (like-photo
      [(input $input)]
      (photo id)
      (user id)))))

(defmethod mutation :undislike-photo [_]
  `((mutation
     undislike-photo
     [($input UndislikePhotoInput!)]
     (undislike-photo
      [(input $input)]
      (photo id)
      (user id)))))

(defmethod mutation :unlike-photo [_]
  `((mutation
     unlike-photo
     [($input UnlikePhotoInput!)]
     (unlike-photo
      [(input $input)]
      (photo id)
      (user id)))))

(defmethod mutation :reset-password [_]
  `((mutation
     reset-password
     [($input ResetPasswordInput!)]
     (reset-password
      [(input $input)]
      (email
       id address
       (user id))))))

(defmethod mutation :signup [_]
  `((mutation
     signup
     [($input SignupInput!)]
     (signup
      [(input $input)]
      auth-token
      (user id username)))))

(defmethod mutation :signin [_]
  `((mutation
     signin
     [($input SigninInput!)]
     (signin
      [(input $input)]
      auth-token
      (user id username)))))

(defmethod mutation :validate-signup [_]
  `((mutation
     validate-signup
     [($input SignupInput!)]
     (validate-signup
      [(input $input)]))))

(defn mutate! [client operation & [params]]
  (let [query (mutation operation)
        result (-send client query params)]
    (get-in result [:data operation])))

(defn mutate [client operation & [params]]
  (let [client (assoc client :throw-exceptions false)
        query (mutation operation)
        result (-send client query params)]
    (->> (get-in result [:data operation])
         (assoc result :data))))

(defn oauth-connect!
  [client provider & [{:keys [referer]}]]
  (-> (merge (into {} client)
             {:method :get
              :headers {"referer" referer}
              :redirect-strategy :none
              :throw-exceptions false
              :uri (routes/path :oauth-connect :name (name provider))})
      (http/request)
      (get-in [:headers "Location"])
      (parse-url)))

(defn oauth-callback! [client provider & [params]]
  (-> (merge (into {} client)
             {:method :get
              :query-params params
              :redirect-strategy :none
              :throw-exceptions false
              :uri (routes/path :oauth-callback :name (name provider))})
      (http/request)
      (get-in [:headers "Location"])
      (parse-url)))
