(ns burningswell.api.users
  "The Burning Swell users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.roles :as roles]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn user-not-found
  "Return a 404 response for a user that could not be found by `id`."
  [id]
  (not-found (format "User %s not found" id)))

(defnk $me$GET
  "Return the currently logged in user."
  {:responses {200 User 404 NotFound}}
  [[:request
    identity :- User]
   [:resources api-client db]]
  (if-let [user (users/by-id db (:id identity))]
    (ok (hal/link api-client :user user))
    (user-not-found (:id identity))))

(defnk $GET
  "List all users."
  {:responses {200 [User]}}
  [[:request
    identity :- User
    query-params :- PaginationParams]
   [:resources api-client db]]
  (->> (users/all db query-params)
       (map #(users/safe-user identity %))
       (hal/links api-client :user)
       (ok)))

(defnk $POST
  "Create a new user."
  {:responses {201 User}}
  [[:request body :- s/Any]
   [:resources api-client db broker]]
  (s/validate (CreateUser db) body)
  (let [user (users/insert db body)
        user (hal/link api-client :user user)]
    (users/add-to-role db user (roles/surfer db))
    (publish broker "users.created" user)
    (created user)))

(defnk $:id$GET
  "Show a user."
  {:responses {200 User 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- User
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [user (users/by-id db id)]
    (->> (users/safe-user identity user)
         (hal/link api-client :user)
         (ok))
    (user-not-found id)))

(defnk $:id$DELETE
  "Delete a user."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [user (users/by-id db id)]
    (do (users/delete db user)
        (publish broker "users.deleted" user)
        (no-content))
    (user-not-found id)))

(defnk $:id$PUT
  "Update a user."
  {:responses {200 User 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-user]
   [:resources api-client db broker]]
  (if-let [user (users/by-id db id)]
    (let [user (users/update db (merge user body))
          user (hal/link api-client :user user)]
      (publish broker "users.updated" user)
      (ok user))
    (user-not-found id)))
