(ns burningswell.api.roles
  "The roles of users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.roles :as roles]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn role-not-found
  "Return a 404 response for a role that could not be found by `id`."
  [id]
  (not-found (format "Role %s not found" id)))

(defnk $GET
  "List all roles."
  {:responses {200 [Role]}}
  [[:request query-params :- PaginationParams]
   [:resources db]]
  (let [roles (roles/roles db query-params)]
    (ok roles)))

(defnk $POST
  "Create a new role."
  {:responses {201 Role}}
  [[:request body :- create-role]
   [:resources db broker]]
  (let [role (roles/create-role db body)]
    (publish broker "roles.created" role)
    (created role)))

(defnk $:id$GET
  "Show a role."
  {:responses {200 Role 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [role (roles/role-by-id db id)]
    (ok role)
    (role-not-found id)))

(defnk $:id$DELETE
  "Delete a role."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [role (roles/role-by-id db id)]
    (do (roles/delete-role db role)
        (publish broker "roles.deleted" role)
        (no-content))
    (role-not-found id)))

(defnk $:id$PUT
  "Update a role."
  {:responses {200 Role 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-role]
   [:resources db broker]]
  (if-let [role (roles/role-by-id db id)]
    (let [role (roles/update-role db (merge role body))]
      (publish broker "roles.updated" role)
      (ok role))
    (role-not-found id)))

(set! *warn-on-reflection* false)
