(ns burningswell.api.ratings
  "The ratings in the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.ratings :as ratings]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn rating-not-found
  "Return a 404 response for a rating that could not be found by `id`."
  [id]
  (not-found (format "Rating %s not found" id)))

(defnk $GET
  "List all ratings."
  {:responses {200 [Rating]}}
  [[:request query-params :- PaginationParams]
   [:resources db]]
  (let [ratings (ratings/ratings db query-params)]
    (ok ratings)))

(defnk $POST
  "Create a new rating."
  {:responses {201 Rating}}
  [[:request
    body :- create-rating
    identity :- User]
   [:resources db broker]]
  (let [rating (ratings/create-rating
                db (assoc body :user-id (:id identity)))]
    (publish broker "ratings.created" rating)
    (created rating)))

(defnk $:id$GET
  "Show a rating."
  {:responses {200 Rating 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [rating (ratings/rating-by-id db id)]
    (ok rating)
    (rating-not-found id)))

(defnk $:id$DELETE
  "Delete a rating."
  {:responses {204 s/Any 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [rating (ratings/rating-by-id db id)]
    (do (ratings/delete-rating db rating)
        (publish broker "ratings.deleted" rating)
        (no-content))
    (rating-not-found id)))

(defnk $:id$PUT
  "Update a rating."
  {:responses {200 Rating 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-rating]
   [:resources db broker]]
  (if-let [rating (ratings/rating-by-id db id)]
    (let [rating (ratings/save-rating db (merge rating body))]
      (publish broker "ratings.updated" rating)
      (ok rating))
    (rating-not-found id)))
