(ns burningswell.api.handler
  (:require [burningswell.api.addresses :as addresses]
            [burningswell.api.airports :as airports]
            [burningswell.api.authentication :refer [wrap-authentication]]
            [burningswell.api.authorization :refer [wrap-authorization]]
            [burningswell.api.coerce :as coerce]
            [burningswell.api.comments :as comments]
            [burningswell.api.config]
            [burningswell.api.continents :as continents]
            [burningswell.api.countries :as countries]
            [burningswell.api.facebook]
            [burningswell.api.jws]
            [burningswell.api.om :as om]
            [burningswell.api.photos :as photos]
            [burningswell.api.ports :as ports]
            [burningswell.api.ratings :as ratings]
            [burningswell.api.regions :as regions]
            [burningswell.api.ring :as ring]
            [burningswell.api.roles :as roles]
            [burningswell.api.root :as root]
            [burningswell.api.schemas :as schemas]
            [burningswell.api.search :as search]
            [burningswell.api.sessions :as sessions]
            [burningswell.api.spots :as spots]
            [burningswell.api.time-zones]
            [burningswell.api.users :as users]
            [burningswell.api.weather.datasets :as datasets]
            [burningswell.api.weather.models :as models]
            [burningswell.api.weather.variables :as variables]
            [fnhouse.docs :as docs]
            [fnhouse.handlers :as handlers]
            [fnhouse.middleware :as middleware]
            [fnhouse.routes :as routes]
            [plumbing.core :refer :all]))

(defn custom-coercion-middleware
  "Wrap a handler with the schema coercing middleware"
  [handler]
  (middleware/coercion-middleware handler coerce/input coerce/output))

(defn attach-docs [resources prefix->ns-sym]
  (let [proto-handlers (handlers/nss->proto-handlers prefix->ns-sym)
        all-docs (docs/all-docs (map :info proto-handlers))]
    (-> resources
        (assoc :api-docs all-docs)
        ((handlers/curry-resources proto-handlers)))))

(def handlers
  {"" 'burningswell.api.root
   "addresses" 'burningswell.api.addresses
   "airports" 'burningswell.api.airports
   "api" 'burningswell.api.om
   "auth/jws" 'burningswell.api.jws
   "comments" 'burningswell.api.comments
   "config" 'burningswell.api.config
   "continents" 'burningswell.api.continents
   "countries" 'burningswell.api.countries
   "oauth/facebook" 'burningswell.api.facebook
   "photos" 'burningswell.api.photos
   "ports" 'burningswell.api.ports
   "ratings" 'burningswell.api.ratings
   "regions" 'burningswell.api.regions
   "roles" 'burningswell.api.roles
   "search" 'burningswell.api.search
   "sessions" 'burningswell.api.sessions
   "spots" 'burningswell.api.spots
   "surfers" 'burningswell.api.users
   "time-zones" 'burningswell.api.time-zones
   "weather/datasets" 'burningswell.api.weather.datasets
   "weather/models" 'burningswell.api.weather.models
   "weather/variables" 'burningswell.api.weather.variables})

(defn wrap-resources
  "Return a handler that adds `resources` to the :resources key of a request."
  [handler resources]
  (fn [request]
    (handler (assoc request :resources resources))))

(defn wrapped-root-handler
  "Take the resources, partially apply them to 'handlers, wrap each
  with a custom coercing middleware, and then compile them into a root
  handler that will route requests to the appropriate underlying
  handler.  Then, wrap the root handler in some standard ring
  middleware."
  [resources]
  (-> (map custom-coercion-middleware (attach-docs resources handlers))
      (routes/root-handler)
      (wrap-authorization)
      (wrap-authentication resources)
      (wrap-resources resources)
      (ring/ring-middleware resources)))
