(ns burningswell.api.system
  (:require [burningswell.api.client :as api]
            [burningswell.api.handler :refer [wrapped-root-handler]]
            [burningswell.api.topology :refer [new-topology]]
            [burningswell.db.connection :refer [new-db]]
            [burningswell.http.server :refer [http-server]]
            [burningswell.namespace :refer [new-reload]]
            [burningswell.rabbitmq.core :refer [rabbitmq]]
            [burningswell.transit :as transit]
            [com.stuartsierra.component :as component]))

(defn- new-api [config]
  (-> (assoc config :handler-fn wrapped-root-handler)
      (http-server)))

(defn- new-api-client [config]
  (api/new-client
   (assoc config
          :content-type :transit+json
          :transit-opts {:handlers transit/write-handlers})))

(defn new-system
  "Return a new API system."
  [config]
  (let [{:keys [api-client broker db reload]} config]
    (-> (component/system-map
         :api (new-api config)
         :api-client (new-api-client api-client)
         :broker (rabbitmq broker)
         :config config
         :db (new-db db)
         :reload (new-reload reload)
         :topology (new-topology {}))
        (component/system-using
         {:api [:api-client :broker :config :db :topology]
          :topology [:broker]}))))

(defmacro with-system
  "Evaluate `body` within the context of a running API system."
  [[symbol config] & body]
  `(let [system# (new-system ~config)
         system# (component/start system#)
         ~symbol system#]
     (try ~@body
          (finally (component/stop system#)))))
