(ns burningswell.api.sessions
  "The surf session at surf spots."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.sessions :as sessions]
            [burningswell.db.spots :as spots]
            [burningswell.http.response :refer [created ok]]
            [clj-time.core :refer [hours minus now]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(defn- session-not-found
  "Return a 404 response for a session that could not be found by `id`."
  [id]
  (not-found (format "Session %s not found" id)))

(defnk $GET
  "List all sessions."
  {:responses {200 [Session]}}
  [[:request query-params :- PaginationParams]
   [:resources db]]
  (let [sessions (sessions/sessions db query-params)]
    (ok sessions)))

(defnk $POST
  "Create a surf session for a spot."
  {:responses {201 Session}}
  [[:request
    body :- create-session
    identity :- User]
   [:resources db broker]]
  (when-let [spot (spots/by-id db (:spot-id body))]
    (let [started-at (or (:started-at body) (now))
          stopped-at (or (:stopped-at body) (minus started-at (hours 1)))
          session (sessions/insert-session
                   db {:user-id (:id identity)
                       :spot-id (:id spot)
                       :started-at started-at
                       :stopped-at stopped-at
                       :rating (:rating body)})]
      (publish broker "sessions.created" session)
      (created session))))

(defnk $:id$GET
  "Show a session."
  {:responses {200 Session 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [session (sessions/session-by-id db id)]
    (ok session)
    (session-not-found id)))

(defnk $:id$DELETE
  "Delete a session."
  {:responses {204 s/Any 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [session (sessions/session-by-id db id)]
    (do (sessions/delete-session db session)
        (publish broker "sessions.deleted" session)
        (no-content))
    (session-not-found id)))

(defnk $:id$PUT
  "Update a session."
  {:responses {200 Session 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-session]
   [:resources db broker]]
  (if-let [session (sessions/session-by-id db id)]
    (let [session (sessions/update-session db (merge session body))]
      (publish broker "sessions.updated" session)
      (ok session))
    (session-not-found id)))
