(ns burningswell.api.photos
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.countries :as countries]
            [burningswell.db.photos :as photos]
            [burningswell.db.images :as images]
            [burningswell.db.regions :as regions]
            [burningswell.http.response :refer [created ok]]
            [no.en.core :refer [format-url]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(defn make-photo [api-client photo]
  (if photo
    (hal/link api-client :photo photo)))

(defn- photo-not-found
  "Return a 404 response for a photo that could not be found by `id`."
  [id]
  (not-found (format "Photo %s not found" id)))

(defnk $GET
  "List all photos."
  {:responses {200 [Photo]}}
  [[:request query-params :- PaginationParams]
   [:resources api-client db]]
  (let [photos (photos/with-images db query-params)
        photos (hal/links api-client :photo photos)]
    (ok photos)))

(defnk $:id$GET
  "Show a photo."
  {:responses {200 Photo 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- User]
   [:resources api-client db]]
  (if-let [photo (photos/with-images-by-id db id {:user identity})]
    (let [photo (hal/link api-client :photo photo)]
      (ok photo))
    (photo-not-found id)))

(defnk $POST
  "Create a new photo."
  {:responses {201 Photo}}
  [[:request
    body :- create-photo
    identity :- User]
   [:resources api-client db broker]]
  (let [body (assoc-in body [:_embedded :user] identity)
        photo (photos/insert db body)
        photo (hal/link api-client :photo photo)]
    (publish broker "photos.created" photo)
    (created photo)))

(defnk $:id$likes$GET
  "Return the likes and dislikes of a photo."
  {:responses {200 [PhotoLike] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- User
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [photo (photos/by-id db id query-params)]
    (ok (photos/likes db photo))
    (photo-not-found id)))

(defnk $:id$likes$POST
  "Like a photo."
  {:responses {200 Photo 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- User]
   [:resources api-client db]]
  (if-let [photo (photos/by-id db id)]
    (do (photos/like db photo identity)
        (->> (photos/with-images-by-id db id {:user identity})
             (hal/link api-client :photo)
             (ok )))
    (photo-not-found id)))

(defnk $:id$likes$DELETE
  "Dislike a photo."
  {:responses {200 Photo 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- User]
   [:resources api-client db]]
  (if-let [photo (photos/by-id db id)]
    (do (photos/dislike db photo identity)
        (->> (photos/with-images-by-id db id {:user identity})
             (hal/link api-client :photo)
             (ok )))
    (photo-not-found id)))
