(ns burningswell.api.addresses
  "The addresses of users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.addresses :as addresses]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn address-not-found
  "Return a 404 response for a address that could not be found by `id`."
  [id]
  (not-found (format "Address %s not found" id)))

(defnk $GET
  "List all addresses."
  {:responses {200 [Address]}}
  [[:request query-params :- PaginationParams]
   [:resources api-client db]]
  (let [addresses (addresses/all db query-params)
        addresses (hal/links api-client :address addresses)]
    (ok addresses)))

(defnk $POST
  "Create a new address."
  {:responses {201 Address}}
  [[:request
    body :- create-address
    identity :- User]
   [:resources api-client db broker]]
  (let [body (assoc-in body [:_embedded :user] identity)
        address (addresses/insert db body)
        address (hal/link api-client :address address)]
    (publish broker "addresses.created" address)
    (created address)))

(defnk $:id$GET
  "Show a address."
  {:responses {200 Address 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [address (addresses/by-id db id)]
    (ok (hal/link api-client :address address))
    (address-not-found id)))

(defnk $:id$DELETE
  "Delete a address."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [address (addresses/by-id db id)]
    (do (addresses/delete db address)
        (publish broker "addresses.deleted" address)
        (no-content))
    (address-not-found id)))

(defnk $:id$PUT
  "Update a address."
  {:responses {200 Address 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-address
    identity :- User]
   [:resources api-client db broker]]
  (if-let [address (addresses/by-id db id)]
    (let [body (-> (assoc body :id id)
                   (assoc-in [:_embedded :user] identity))
          address (addresses/update db (merge address body))
          address (hal/link api-client :address address)]
      (publish broker "addresses.updated" address)
      (ok address))
    (address-not-found id)))

(set! *warn-on-reflection* false)
