(ns burningswell.api.slugs
  (:require [claro.data :as data]
            [clojure.string :as str]
            [no.en.core :refer [format-url]]
            [inflections.core :as inf]))

(defn- sanitize
  "Convert special characters to dashes."
  [s]
  (some-> s (str/replace #"[\s\.\-/_]+" "-")))

(defn- parameterize [& args]
  (->> (map str args)
       (remove str/blank?)
       (str/join "-" )
       (sanitize)))

(defmulti slug-name
  "Returns the slug name for the `entity` of `type`."
  (fn [type entity] type))

(defmulti slug-path
  "Returns the slug path for the `entity` of `type`."
  (fn [type entity] type))

(defn slug-url
  "Returns the slug path for the `entity` of `type`."
  [env type entity]
  (->> (slug-path type entity)
       ;; TODO: Use web client
       (assoc (:client env) :uri)
       (format-url)))

(defrecord Slug [type entity]
  data/Cost
  (cost [resolvable batch] 0)
  data/Resolvable
  (resolve! [params env]
    {:name (slug-name type entity)
     :path (slug-path type entity)
     :url (slug-url env type entity)}))

(defmethod slug-name :photo
  [type {:keys [id title]}]
  (parameterize id title))

(defmethod slug-name :user
  [type {:keys [id username]}]
  (parameterize id username))

(defmethod slug-name :weather-model
  [type {:keys [id name description]}]
  (parameterize id (str/upper-case name) description))

(defmethod slug-name :default
  [type {:keys [id name]}]
  (assert id (str "Not slug name defined for type:" type))
  (parameterize id name))

(defmethod slug-path :default [type entity]
  (str "/" (-> type name inf/plural (str/replace #"-" "/"))
       "/" (slug-name type entity)))
