(ns burningswell.api.oauth.users
  (:require [cheshire.core :as json]
            [datumbazo.core :as sql]
            [datumbazo.io :refer [citext]]))

(defn- encode-profile [profile]
  `(cast ~(json/generate-string profile) :jsonb))

(defmulti normalize
  "Normalize the `profile` of the given `provider`."
  (fn [provider profile] (-> provider :name keyword)))

(defmethod normalize :google
  [provider {:keys [id name emails] :as profile}]
  {:emails (for [{:keys [value]} emails] {:address value})
   :first-name (:given-name name)
   :id id
   :last-name (:family-name name)
   :profile profile})

(defmethod normalize :facebook
  [provider {:keys [id first-name last-name email] :as profile}]
  {:emails (when email [{:address email}])
   :first-name first-name
   :id id
   :last-name last-name
   :profile profile})

(defmethod normalize :linkedin
  [provider {:keys [id email-address first-name last-name] :as profile}]
  {:emails (when email-address [{:address email-address}])
   :first-name first-name
   :id id
   :last-name last-name
   :profile profile})

(defmethod normalize :twitter
  [provider {:keys [id-str] :as profile}]
  {:id id-str
   :profile profile})

(defn- provider-table [provider]
  (keyword (str (:name provider) ".users")))

(defn- insert-user! [db profile]
  (first @(sql/insert db :users [:first-name :last-name]
            (sql/values [(select-keys profile [:first-name :last-name])])
            (sql/returning :*))))

(defn- save-profile! [db provider user profile]
  @(sql/insert db (provider-table provider) []
     (sql/values [{:id (:id profile)
                   :profile (encode-profile (:profile profile))
                   :user-id (:id user)}])
     (sql/on-conflict [:id]
       (sql/do-update {:profile (encode-profile (:profile profile))}))))

(defn- save-emails! [db user profile]
  (when-not (empty? (:emails profile))
    @(sql/insert db :emails [:address :user-id :verified-at]
       (sql/values (for [{:keys [address]} (:emails profile)]
                     {:address (citext address)
                      :user-id (:id user)
                      :verified-at `(clock_timestamp)}))
       (sql/on-conflict [:address]
         (sql/do-update {:user-id :EXCLUDED.user-id})))))

(defn- user-by-id [db provider profile]
  (first @(sql/select db [:users.*]
            (sql/from :users)
            (sql/join (sql/as (provider-table provider) :providers)
                      '(on (= :users.id :providers.user-id)))
            (sql/where `(= :providers.id ~(:id profile))))))

(defn- user-by-emails [db profile]
  (first @(sql/select db [:users.*]
            (sql/from :users)
            (sql/join :emails.user-id :users.id)
            (sql/where `(in :address ~(map :address (:emails profile)))))))

(defn insert! [db provider profile]
  (let [profile (normalize provider profile)]
    (sql/with-transaction [db db]
      (let [user (insert-user! db profile)]
        (save-emails! db user profile)
        (save-profile! db provider user profile)
        user))))

(defn update! [db provider profile]
  (let [profile (normalize provider profile)]
    (sql/with-transaction [db db]
      (when-let [user (or (user-by-id db provider profile)
                          (user-by-emails db profile))]
        (save-emails! db user profile)
        (save-profile! db provider user profile)
        user))))
