(ns burningswell.api.nodes
  (:require [burningswell.api.airports :as airports]
            [burningswell.api.continents :as continents]
            [burningswell.api.countries :as countries]
            [burningswell.api.images :as images]
            [burningswell.api.photos :as photos]
            [burningswell.api.ports :as ports]
            [burningswell.api.regions :as regions]
            [burningswell.api.spots :as spots]
            [burningswell.api.time-zones :as time-zones]
            [burningswell.api.users :as users]
            [burningswell.api.weather.models :as weather-models]
            [burningswell.api.weather.variables :as weather-variables]
            [burningswell.routes :as routes]
            [claro.data :as data]
            [clojure.edn :as edn]
            [ring.util.codec :as codec]))

(defn- parse-id
  "Parse `s` as a Base64 encoded node id."
  [s]
  (edn/read-string (String. (codec/base64-decode s))))

(defmulti by-type
  "Returns the node of the given :type and :id in `params`."
  (fn [params] (:type params)))

(defmethod by-type :airport [params]
  (airports/map->Airport params))

(defmethod by-type :continent [params]
  (continents/map->Continent params))

(defmethod by-type :country [params]
  (countries/map->Country params))

(defmethod by-type :image [params]
  (images/map->Image params))

(defmethod by-type :photo [params]
  (photos/map->Photo params))

(defmethod by-type :port [params]
  (ports/map->Port params))

(defmethod by-type :region [params]
  (regions/map->Region params))

(defmethod by-type :spot [params]
  (spots/map->Spot params))

(defmethod by-type :time-zone [params]
  (time-zones/map->TimeZone params))

(defmethod by-type :user [params]
  (users/map->User params))

(defmethod by-type :weather-model [params]
  (weather-models/map->Model params))

(defmethod by-type :weather-variable [params]
  (weather-variables/map->Variable params))

(defmethod by-type :default [_] nil)

(defn by-slug
  "Returns the node for the given `slug`."
  [slug]
  (when-let [{:keys [handler route-params]} (routes/match slug)]
    (when-let [id (-> route-params :id Long/parseLong)]
      (by-type {:type handler :id id}))))

(defrecord Node [id slug]
  data/Resolvable
  (resolve! [_ _]
    (cond
      (string? id)
      (by-type (parse-id id))
      (string? slug)
      (by-slug slug)
      :else
      (throw (ex-info (str "Invalid id:" id)
                      {:type ::invalid-id
                       :id id})))))
