(ns burningswell.api.graphql
  (:require [alumbra.analyzer :as analyzer]
            [alumbra.claro :as claro]
            [alumbra.core :as alumbra]
            [alumbra.parser :as parser]
            [alumbra.pipeline :as pipeline]
            [alumbra.validator :as validator]
            [alumbra.web.graphiql-workspace :as graphiql-workspace]
            [burningswell.api.addresses :as addresses]
            [burningswell.api.airports :as airports]
            [burningswell.api.continents :as continents]
            [burningswell.api.countries :as countries]
            [burningswell.api.dislikes :as dislikes]
            [burningswell.api.emails :as emails]
            [burningswell.api.images :as images]
            [burningswell.api.likes :as likes]
            [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.middleware.exceptions :as exceptions]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.middleware.transaction :as transaction]
            [burningswell.api.nodes :as nodes]
            [burningswell.api.oauth.providers :as oauth-providers]
            [burningswell.api.page-views :as page-views]
            [burningswell.api.photos :as photos]
            [burningswell.api.ports :as ports]
            [burningswell.api.regions :as regions]
            [burningswell.api.roles :as roles]
            [burningswell.api.search :as search]
            [burningswell.api.signin :as signin]
            [burningswell.api.signup :as signup]
            [burningswell.api.spots :as spots]
            [burningswell.api.time-zones :as time-zones]
            [burningswell.api.users :as users]
            [burningswell.api.viewer :as viewer]
            [burningswell.api.weather :as weather]
            [burningswell.api.weather.datasources :as datasources]
            [burningswell.api.weather.models :as models]
            [burningswell.api.weather.variables :as variables]
            [claro.access :as access]
            [claro.data :as data]
            [claro.engine :as engine]
            [claro.middleware.observe :as observe]
            [clj-time.coerce :as time]
            [clojure.java.io :as io]
            [no.en.core :refer [format-url]]))

(require 'burningswell.api.transform)

(def Mutation
  {:create-page-view (page-views/map->Create {})
   :create-spot (spots/map->Create {})
   :dislike-photo (dislikes/map->DislikePhoto {})
   :like-photo (likes/map->LikePhoto {})
   :signin (signin/map->Create {})
   :signup (signup/map->Create {})
   :undislike-photo (dislikes/map->UndislikePhoto {})
   :unlike-photo (likes/map->UnlikePhoto {})})

(def Query
  {:address (addresses/map->Address {})
   :addresses (addresses/map->Addresses {})
   :airport (airports/map->Airport {})
   :airports (airports/map->Airports {})
   :continent (continents/map->Continent {})
   :continents (continents/map->Continents {})
   :countries (countries/map->Countries {})
   :country (countries/map->Country {})
   :email (emails/map->Email {})
   :emails (emails/map->Emails {})
   :image (images/map->Image {})
   :images (images/map->Images {})
   :node (nodes/map->Node {})
   :oauth-provider (oauth-providers/map->Provider {})
   :oauth-providers (oauth-providers/map->Providers {})
   :photo (photos/map->Photo {})
   :photos (photos/map->Photos {})
   :port (ports/map->Port {})
   :ports (ports/map->Ports {})
   :region (regions/map->Region {})
   :regions (regions/map->Regions {})
   :role (roles/map->Role {})
   :roles (roles/map->Roles {})
   :search (search/map->Search {})
   :spot (spots/map->Spot {})
   :spots (spots/map->Spots {})
   :time-zone (time-zones/map->TimeZone {})
   :time-zones (time-zones/map->TimeZones {})
   :user (users/map->User {})
   :users (users/map->Users {})
   :viewer (viewer/map->Viewer {})
   :weather (weather/map->Weather {})
   :weather-datasource (datasources/map->Datasource {})
   :weather-datasources (datasources/map->Datasources {})
   :weather-model (models/map->Model {})
   :weather-models (models/map->Models {})
   :weather-variable (variables/map->Variable {})
   :weather-variables (variables/map->Variables {})
   :wave-heights (variables/map->WaveHeights {})
   :wind-speeds (variables/map->WindSpeeds {})})

(defn schema
  "Returns the GraphQL schema as a string."
  []
  (slurp (io/resource "burningswell.graphql")))

(defn- generate-context
  [{:keys [headers] :as request}]
  {:user (:user request)
   :user-agent (get headers "user-agent")})

(defn wrap-observe-duration [engine]
  (->> (fn [result delta]
         (binding [*print-length* 5]
           ;; (clojure.pprint/pprint result)
           (println (format "%.2f ms" (/ delta 1000000.0))))
         result)
       (observe/wrap-observe-duration engine)))

(defn- engine
  "Returns the claro engine."
  [{:keys [publisher] :as env}]
  (-> (engine/engine {:max-cost 90})

      (identifier/wrap-output)
      (access/wrap-access)
      ;; (wrap-observe-duration)

      (events/wrap-publish-events publisher)
      (events/wrap-events)

      (commands/wrap-publish-commands publisher)
      (commands/wrap-commands)
      (conform/wrap-conform-params)
      (transaction/wrap-transaction)
      (exceptions/wrap-exceptions)))

(def scalars
  "The custom scalar types used in the GraphQL schema."
  {"Time" {:encode time/to-string :decode time/to-date}
   "UUID" {:encode str :decode #(java.util.UUID/fromString %)}
   "Url" {:encode str :decode #(java.net.URL. %)}})

(defn graphql-handler-opts
  "Returns the GraphQL handler options for `env`."
  [env]
  {:context-fn #'generate-context
   :engine (engine env)
   :env env
   :mutation Mutation
   :query Query
   :scalars scalars
   :schema (schema)})

(defn graphql-handler
  "Returns the GraphQL handler for `env`."
  [env]
  (let [{:keys [schema
                query mutation subscription
                engine env context-fn
                scalars directives]
         :as opts} (graphql-handler-opts env)
        schema (#'alumbra/analyze schema)
        opts (assoc opts :schema schema)]
    (->> {:parser-fn #(parser/parse-document %)
          :validator-fn (validator/validator schema)
          :canonicalize-fn (analyzer/canonicalizer schema)
          :executor-fn (claro/executor opts)}
         (merge opts)
         (#'pipeline/make-graphql-handler))))

(defn graphql-url
  "Returns the GraphQL url."
  [client]
  (format-url (assoc client :uri "/graphql")))

(defn graphiql-workspace
  "Returns the GraphiQL workspace handler."
  [{:keys [client] :as env}]
  (graphiql-workspace/handler
   {:config
    {:defaultUrl (graphql-url client)
     :defaultQuery ""}
    :title "Burning Swell GraphiQL"}))

(defn- parse-schema
  [schema]
  (analyzer/analyze-schema schema parser/parse-schema))

(defn parse-document [schema document]
  (let [analyzed-schema (parse-schema schema)]
    (->> (parser/parse-document document)
         (analyzer/canonicalize-operation analyzed-schema))))

(comment

  (clojure.pprint/pprint
   (parse-schema (schema)))

  (spit "SCHEMA.edn" (parse-schema (schema)))

  (clojure.pprint/pprint
   (parse-document (schema) "{ continents { edges { node { id } } } }"))

  (clojure.pprint/pprint
   (data/resolve-batch!
    (map->TimeZone {:id 1})
    {:db (-> reloaded.repl/system :db)}))

  (clojure.pprint/pprint
   (data/resolve!
    (map->WaveHeights
     {:latitude 43.40921730553149
      :longitude -2.69500976358495
      :start #inst "2017-02-13T00:00:00"
      :end #inst "2017-02-13T12:00:00"})
    {:db (-> reloaded.repl/system :db)}))

  )
