(ns burningswell.api.gen
  (:require [clojure.spec.alpha :as s]
            [sqlingvo.expr :as expr]
            [datumbazo.gen :as db-gen]
            [clojure.test.check.generators :as gen]
            [burningswell.api.core :as core]
            [datumbazo.core :as sql]
            [inflections.core :as infl]
            [clojure.string :as str]))

(def not-empty-string
  (s/gen (s/and string? not-empty)))

(def email
  (gen/fmap
   (fn [[user domain top-level]]
     (str user "@" domain "." top-level))
   (gen/tuple not-empty-string
              not-empty-string
              not-empty-string)))

(defn password [env]
  gen/string-alphanumeric)

(defn username [env]
  gen/string-alphanumeric)

(defn signup-input [env]
  (gen/hash-map
   :email email
   :password (password env)
   :username (username env)))

(defn signup [env]
  (gen/hash-map :input (signup-input env)))

(defn login [env]
  (gen/hash-map
   :email email
   :password (password env)
   :username (username env)))

(defn- db-column [env column]
  (let [column (expr/parse-column column)]
    (->> @(sql/select (:db env) [(:name column)]
            (sql/from (:table column))
            (sql/limit 20))
         (map (:name column))
         (gen/elements))))

(defn graphql-id [env type]
  (let [table (-> type name infl/plural)]
    (->> (gen/hash-map
          ;; Key order matters for base64 ids :/
          :id (db-column env (keyword (str table ".id")))
          :type (gen/return type))
         (gen/fmap core/base64-encode))))

(defn- input [input]
  (gen/hash-map :input input))

(defn generate
  "Like `gen/generate`, but with `core/*env` bound to `env`."
  [env generator & [size]]
  (core/with-env env (gen/generate generator)))

(defn sample
  "Like `gen/sample`, but with `core/*env` bound to `env`."
  [env generator & [num-samples]]
  (core/with-env env (gen/sample generator (or num-samples 10))))

(defn id
  "Generate GraphQL ids fo the given `type`."
  [env type]
  (graphql-id env type))

(defmulti mutation
  (fn [env name] (keyword name)))

(defmethod mutation :create-page-view [env _]
  (s/gen :burningswell.api.page-views/create))

(defmethod mutation :create-spot [env _]
  (s/gen :burningswell.api.spots/create))

(defmethod mutation :dislike-photo [env _]
  (input (gen/hash-map :photo-id (graphql-id env :photo))))

(defmethod mutation :like-photo [env _]
  (input (gen/hash-map :photo-id (graphql-id env :photo))))
