(ns burningswell.api.errors
  (:require [burningswell.api.core :as core]
            [burningswell.api.validation :as v]
            [clojure.spec.alpha :as s]
            [phrase.alpha :refer [defphraser phrase]]))

(defphraser (v/max-length max-length)
  [_ _ max-length]
  (format "is too short (max. %s chars)" max-length))

(defphraser (v/min-length min-length)
  [_ _ min-length]
  (format "is too short (min. %s chars)" min-length))

(defphraser (v/db-id-exists? table)
  [_ _ table]
  (format "does not exist"))

(defphraser (s/conformer burningswell.api.specs/conform-id)
  [_ _]
  (format "is not a valid identifier"))

(defphraser v/email-available?
  [_ _] "has already been taken")

(defphraser v/email-valid?
  [_ _] "is not valid")

(defphraser v/username-available?
  [_ _] "has already been taken")

(defn messages [problems]
  (reduce (fn [errors {:keys [in msg] :as problem}]
            (if (and msg (not-empty in))
              (update-in errors in #(conj (or % []) (:msg problem)))
              errors))
          {} problems))

(defn- assoc-message [env problem]
  (assoc problem :msg (phrase env problem)))

(defn- assoc-messages [env problems]
  (mapv #(assoc-message env %) problems))

(defn update-problems [env explain-data]
  (update explain-data ::s/problems (partial assoc-messages env)))

(defn explain-data [env spec data]
  (->> (core/explain-data env spec data)
       (update-problems env)))

(defn explain-errors [env spec data]
  (->> (explain-data env spec data) ::s/problems messages))

(comment

  (clojure.pprint/pprint
   (explain-data
    reloaded.repl/system
    :burningswell.api/signup
    {:input
     {:email "bodhi@example.com"
      :username "bodhi"
      :password "x"}}))

  (clojure.pprint/pprint
   (explain-errors
    reloaded.repl/system
    :burningswell.api/signup
    {:input
     {:email "bodhi@example.com"
      :username "bodhi"
      :password "x"}}))

  (clojure.pprint/pprint
   (explain-errors
    reloaded.repl/system
    :burningswell.api/signup
    {:input
     {:email "bodhi2"
      :username "bodhi2"
      :password "xsasasas"}}))

  )
