(ns burningswell.api.emails
  (:require [alumbra.claro :as claro]
            [burningswell.api.core :as core]
            [burningswell.api.middleware.authentication :as auth]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.db.emails :as emails]
            [claro.access :as access]
            [claro.data :as data]
            [clojure.spec.alpha :as s]))

(s/def :burningswell.api.emails/params
  (s/keys :opt-un [:burningswell.api.emails/sort
                   :burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction]))

;; Private email

(defrecord PrivateEmail [id]
  access/Read
  (can-read? [params {:keys [user]} email]
    (or (auth/has-role? user :admin)
        (= (:id user) (:user-id email))))

  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :email :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} emails]
    (emails/select-batch db emails)))

;; Email by id

(defrecord EmailById [id]
  claro/GraphQL
  (__typename [_ _] "Email")

  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :email :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} emails]
    (emails/select-batch db emails)))

;; Email by address

(defrecord EmailByAddress [address]
  claro/GraphQL
  (__typename [_ _] "Email")

  identifier/Identifier
  (identifier [_ _]
    {:type :email :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} emails]
    (emails/select-batch db emails {:join {:address :citext}})))

;; Email

(defrecord Email [id address]
  data/Resolvable
  data/PureResolvable
  (resolve! [resolvable env]
    (cond
      id (map->EmailById {:id id})
      address (map->EmailByAddress {:address address})
      :else nil)))

;; Emails

(defrecord Emails [after before direction first last sort query]
  access/Resolve
  (can-resolve? [params {:keys [user]}]
    (auth/has-role? user :admin))

  conform/Params
  (conform [params env]
    :burningswell.api.emails/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:direction direction
          :limit (core/limit params)
          :offset (core/offset params)
          :query query
          :sort sort}
         (emails/search db))))
