(ns burningswell.api.countries
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.api.weather :as weather]
            [burningswell.db.countries :as countries]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [burningswell.api.gen :as gens]))

(s/def :burningswell.api.countries/id
  (s/and ::specs/id (v/db-id-exists? :countries)))

(s/def :burningswell.api.countries/continents
  (s/nilable (s/coll-of :burningswell.api.continents/id)))

(s/def :burningswell.api.countries/except
  (s/nilable (s/coll-of :burningswell.api.countries/id)))

(s/def :burningswell.api.countries/params
  (s/keys :opt-un [:burningswell.api.countries/continents
                   :burningswell.api.countries/except
                   :burningswell.api.countries/sort
                   :burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/min-spots
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.specs/location]))

(defrecord Country [id name continent]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :country
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} countries]
    (countries/select-batch db countries {:except [:geom]})))

(defrecord Countries [after before continents direction distance except
                      first last location min-spots sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.countries/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:continents continents
          :direction direction
          :distance distance
          :except except
          :limit (core/limit params)
          :location location
          :min-spots min-spots
          :offset (core/offset params)
          :query query
          :sort sort}
         (countries/search db))))

(defrecord Airports [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.airports/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} countries]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :airports params)}
         (sql/has-many db countries :countries :airports))))

(defrecord Photos [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.photos/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} countries]
    (->> {:join-table :photos-countries
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :photos params)}
         (sql/has-and-belongs-to-many
          db countries :countries :photos))))

(defrecord Regions [after before direction first last id min-spots sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.regions/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} countries]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :regions params)
          :where (sql/where `(>= :regions.spot-count ~min-spots))}
         (sql/has-many db countries :countries :regions))))

(defrecord Spots [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.spots/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} countries]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :spots params)}
         (sql/has-many db countries :countries :spots))))

(defrecord Ports [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.ports/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} countries]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :ports params)}
         (sql/has-many db countries :countries :ports))))

(weather/define-weather-resolvers :countries :weather.countries)
