(ns burningswell.api.client
  (:require [cheshire.core :as json]
            [clj-http.client :as http]
            [clojure.pprint :refer [pprint]]
            [clojure.walk :as walk]
            [inflections.core :as inf]
            [venia.core :as v]
            [no.en.core :refer [parse-url]]
            [burningswell.routes :as routes]
            [clojure.string :as str]))

(defprotocol IClient
  (-send [client query variables] "Send the GraphQL `query` to the server."))

(defprotocol IQuery
  (-compile [query] "Compile the GraphQL `query` to a string."))

(def defaults
  "The default client options."
  {:accept "application/json"
   :as :auto
   :coerce :always
   :content-type "application/json"
   :date-format "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"
   :request-method :post
   :scheme :http
   :server-name "localhost"
   :server-port 7000
   :uri "/graphql"})

(defn- underscore-query
  "Transform all dashes in the keywords of `m` to underscores."
  [m]
  (let [transform
        (fn transform [v]
          (cond
            (keyword? v)
            (inf/underscore v)
            (sequential? v)
            (map transform v)
            :else v))
        f (fn [[k v]]
            [(transform k)
             (if (sequential? v)
               (transform v) v)])]
    (walk/postwalk (fn [x] (if (map? x) (into {} (map f x)) x)) m)))

(extend-protocol IQuery
  clojure.lang.PersistentArrayMap
  (-compile [query]
    (-> query underscore-query v/graphql-query))

  String
  (-compile [query]
    query))

;; Client

(defn- parse-response [response]
  (-> response :body inf/hyphenate-keys))

(defrecord CljHttp [scheme server-name server-port uri]
  IClient
  (-send [client query variables]
    (try (->> {:query (-compile query)
               :variables (inf/underscore-keys variables)}
              (json/generate-string)
              (assoc (into {} client) :body)
              (http/request)
              parse-response)
         (catch Exception e
           (let [{:keys [body] :as request} (ex-data e)]
             (pprint body)
             (throw (ex-info (with-out-str (pprint body))
                             (or request {})
                             e)))))))

(defn client
  "Returns a new GraphQL client."
  [config]
  (-> (merge defaults config)
      (map->CljHttp)))

(defn q
  "Send the GraphQL `query` to the server."
  [client query & [variables]]
  (-send client query variables))

(defn qp
  "Send the GraphQL `query` to the server and pretty print the result."
  [client query & [variables]]
  (pprint (-send client query variables)))

(defmulti mutation
  (fn [operation] (keyword operation)))

(defmethod mutation :create-spot [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "create_spot"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :CreateSpotInput!}]
   :venia/queries
   [[:create_spot {:input :$input}
     [[:spot
       [:id :name
        [:user [:id]]]]]]]})

(defmethod mutation :create-page-view [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "create_page_view"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :CreatePageViewInput!}]
   :venia/queries
   [[:create_page_view {:input :$input}
     [:session_id :url
      [:location [:latitude :longitude]]
      [:user [:id]]]]]})

(defmethod mutation :dislike-photo [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "dislike_photo"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :DislikePhotoInput!}]
   :venia/queries
   [[:dislike_photo {:input :$input}
     [[:photo [:id]]
      [:user [:id]]]]]})

(defmethod mutation :like-photo [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "like_photo"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :LikePhotoInput!}]
   :venia/queries
   [[:like_photo {:input :$input}
     [[:photo [:id]]
      [:user [:id]]]]]})

(defmethod mutation :undislike-photo [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "undislike_photo"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :UndislikePhotoInput!}]
   :venia/queries
   [[:undislike_photo {:input :$input}
     [[:photo [:id]]
      [:user [:id]]]]]})

(defmethod mutation :unlike-photo [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "unlike_photo"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :UnlikePhotoInput!}]
   :venia/queries
   [[:unlike_photo {:input :$input}
     [[:photo [:id]]
      [:user [:id]]]]]})

(defmethod mutation :signup [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "signup"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :SignupInput!}]
   :venia/queries
   [[:signup {:input :$input}
     [:auth_token
      [:user [:id :username]]]]]})

(defmethod mutation :signin [_]
  {:venia/operation
   {:operation/type :mutation
    :operation/name "signin"}
   :venia/variables
   [{:variable/name "input"
     :variable/type :SigninInput!}]
   :venia/queries
   [[:signin {:input :$input}
     [:auth_token
      [:user [:id :username]]]]]})

(defn mutate! [client operation & [params]]
  (let [query (v/graphql-query (mutation operation))
        result (-send client query params)]
    (get-in result [:data operation])))

(defn mutate [client operation & [params]]
  (let [client (assoc client :throw-exceptions false)
        query (v/graphql-query (mutation operation))
        result (-send client query params)]
    (->> (get-in result [:data operation])
         (assoc result :data))))

(defn oauth-connect!
  [client provider & [{:keys [referer]}]]
  (-> (merge (into {} client)
             {:method :get
              :headers {"referer" referer}
              :redirect-strategy :none
              :throw-exceptions false
              :uri (routes/path :oauth-connect :name (name provider))})
      (http/request)
      (get-in [:headers "Location"])
      (parse-url)))

(defn oauth-callback! [client provider & [params]]
  (-> (merge (into {} client)
             {:method :get
              :query-params params
              :redirect-strategy :none
              :throw-exceptions false
              :uri (routes/path :oauth-callback :name (name provider))})
      (http/request)
      (get-in [:headers "Location"])
      (parse-url)))
