(ns burningswell.api.airports
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.db.airports :as airports]
            [claro.data :as data]
            [clojure.spec.alpha :as s]))

(s/def :burningswell.api.airports/id
  (s/and ::specs/id (v/db-id-exists? :airports)))

(s/def :burningswell.api.airports/except
  (s/nilable (s/coll-of :burningswell.api.airports/id)))

(s/def :burningswell.api.airports/countries
  (s/nilable (s/coll-of :burningswell.api.countries/id)))

(s/def :burningswell.api.airports/regions
  (s/nilable (s/coll-of :burningswell.api.regions/id)))

(s/def :burningswell.api.airports/params
  (s/keys :opt-un [:burningswell.api.airports/countries
                   :burningswell.api.airports/except
                   :burningswell.api.airports/regions
                   :burningswell.api.airports/sort
                   :burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.specs/location]))

(defrecord Airport [id name country region]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :airport
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} airports]
    (airports/select-batch db airports)))

(defrecord Airports [after before countries distance direction except
                     first last location regions sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.airports/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:countries countries
          :direction direction
          :distance distance
          :except except
          :limit (core/limit params)
          :location location
          :offset (core/offset params)
          :query query
          :regions regions
          :sort sort}
         (airports/search db))))
