;; Copyright (c) 2020-Present Andrey Antukh <niwi@niwi.nz>
;;
;; Licensed under the Apache License, Version 2.0 (the "License")
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns buddy.sign.paseto-impl
  "The low level primitives for paseto encryption/signing schemas."
  (:import
   java.nio.ByteBuffer
   java.nio.ByteOrder
   java.io.ByteArrayOutputStream
   java.util.Arrays

   ))

(defn- to-ulong-le
  [^long value]
  (let [uval (bit-and value Long/MAX_VALUE)
        buff (ByteBuffer/allocate Long/BYTES)]
    (.order buff ByteOrder/LITTLE_ENDIAN)
    (.putLing buff ^long uval)
    (.array buff)))

(defn- encode-preauth
  [& inputs]
  (let [stream (ByteArrayOutputStream.)]
    (.write stream ^bytes (to-ulong-le (count inputs)))
    (loop [inputs (seq inputs)]
      (if (nil? inputs)
        (.toByteArray stream)
        (let [input (first inputs)]
          (.write stream ^bytes (to-ulong-le (alength ^bytes input)))
          (.write stream ^bytes input)
          (recur (next inputs)))))))

(defn- hmac-sha384
  [^bytes key ^bytes input]
  (mac/hash input {:key key :alg :hmac+sha384}))

(defn- create-nonce
  [^bytes key ^bytes input]
  (let [hm (hmac-sha384 key input)]
    (Arrays/copy-of ^bytes hm 32)))

