(ns bss.rampant.sorting)

;; generic
;; ----------------------------

;; FIXME:
;; This does not handle compound keys properly
;; 1. option 1: include compound list in actual list, and support inverting arrays + flatten before sorting
;; 2. option 2: handle lists of keys here, and treat this differently to handling those keys in sequence
(defn update-opt-list
  "Given a list of keys, a new value.
   If value match?-es head of list, toggle the value
   Otherwise remove all that match? key from list, and add key to front"
  [old-state key toggle match?]
  (let [is-key? (partial match? key)
        [head & tail] old-state
        filtered-tail (remove is-key? tail)]
    (if (is-key? head)
      (cons (toggle head) filtered-tail)
      (cons key (cons head filtered-tail)))))

;; helpers
;; ----------------------------

(defn match-sort-keys
  "Relaxed equality check to ignore case"
  [& keys]
  (apply = (map #(.toLowerCase (name %)) keys)))

(defn toggle-case
  "Uppercase to lowercase, and vice versa"
  [s]
  (let [lc (.toLowerCase s)]
    (if (= s lc) (.toUpperCase s) lc)))

(def toggle-keyword-case (comp keyword toggle-case name))

(defn asc?
  "Interpret :ascending and :DESCENDING convention based on keyword case"
  [key]
  (let [n (name key)]
    (= (.toLowerCase n) n)))

(def desc? (comp not asc?))

(defn ->lower
  "Normalize keyword to lowercase (eg. for lookup)"
  [k]
  (keyword (.toLowerCase (name k))))

(defn sym->compare
  "Map :UPPER or :lower keys to associated comparator (see asc? and desc?)"
  [sym]
  (if (asc? sym) compare (comp - compare)))

(defn sort-by-key
  "Sort collection of maps according to particular cased-key"
  [rows sort-key]
  (sort-by (->lower sort-key)
           (sym->compare sort-key)
           rows))

(defn compare-keys
  "Build single-step comparison function over multiple cased-keys"
  [sort-keys]
  #(or (first
         (remove zero?
                 (map (fn [key a b]( (sym->compare key) a b))
                      sort-keys %1 %2)))
       0))


;; Top level sort API
;; ----------------------------

(defn sort-by-keys
  "Sort collection of maps according to list of cased-keys in priority"
  [keys rows]
  (reduce sort-by-key rows (reverse keys)))

(defn update-sort-keys
  "Update sort-keys in response to new key.
   If key is highest priority toggle asc/desc, otherwise set key as highest priority"
  [sort-keys key_s]
  (update-opt-list sort-keys key_s toggle-keyword-case match-sort-keys))

;;;;;;;;;;;; This file autogenerated from src-cljx/bss/rampant/sorting.cljx
