(ns singularity.support

  "Helper methods for the Singularity client.  Many of this may be left over
  from other projects, and need to be cleaned up."
  
  (:use [clojure.data.json :only (Write-JSON write-json)])
  (:require [clojure.string :as str]
            [clojure.data.json :as json])
  (:import
   brilliantarc.singularity.SingularityException
   java.text.SimpleDateFormat
   [java.util Calendar TimeZone]))

(def XML-SCHEMA-DATETIME (SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ssZ"))

(defn- json-str-output
  "Call toString on errant objects in the JSON."
  [x out escape-unicode?] (write-json (str x) out escape-unicode?))

;; Handle Mongo's org.bson.types.ObjectId type
(extend org.bson.types.ObjectId Write-JSON
        {:write-json json-str-output})

(defn wants-json [data & [status]]
  {:status (or status 200)
   :headers {"Content-Type" "application/json; charset=utf-8"}
   :body (json/json-str data)})

(defn wants-xml [data & [status]]
  {:status (or status 200)
   :headers {"Content-Type" "application/xml; charset=utf-8"}
   :body data})

(defmacro with-credentials
  "Bind the credentials to the current thread and include with requests to
  Singularity."
  [credentials & body]
  `(binding [singularity.request/*credentials* ~credentials]
     (do ~@body)))

(defmacro as
  "Bind the credentials to the current thread and include with requests to
  Singularity."
  [credentials & body]
  `(binding [singularity.request/*credentials* ~credentials]
     (do ~@body)))

(defmacro get-or-create
  "This is a pretty typical pattern:  check to see if a node exists, and if not,
  create it.  Simply wrap your calls to both states in get-or-create:

    (get-or-create
      (singularity/definition \"Some Definition\")
      (singularity/create-definition \"Some Definition\"))

  Handles the exception calls for you properly.
  "
  [get-func create-func]
  `(try
     ~get-func
     (catch SingularityException e#
       (if (= (.getStatus e#) 404)
         ~create-func
         (throw e#)))))

(defn failure
  "Throws a SingularityException.  Makes it easy to generate error messages
  that bubble up to the API layer.  For example:

    (failure 404 \"Could not find a portfolio named \" name)
  "
  [status & message]
  (throw (SingularityException. (apply str message) status)))

(defn right-now
  "Return the time for right now, GMT."
  []
  (.getTime (Calendar/getInstance (TimeZone/getTimeZone "GMT"))))

(defn right-now-timestamp
  "Return the time right now as a timestamp."
  []
  (.getTimeInMillis (Calendar/getInstance (TimeZone/getTimeZone "GMT"))))

(defn xml-time
  "Return a date/time string in XML Schema-compatible format for GMT."
  [& [when]]
  (let [when (or when (right-now))]
    (.format XML-SCHEMA-DATETIME when)))

(defn as-int
  "Convert a string to an integer.  If the string is not a valid integer, raises
  a SingularityException.  If the string is blank, returns nil."
  [value]
  (if (and value (not (str/blank? value)))
    (try
      (Integer/parseInt value)
      (catch NumberFormatException e
        (failure 406 "Invalid number format for" value)))
    nil))

(defn as-float
  "Convert a string to a float.  If the string is not a valid float, raises
  a SingularityException.  If the string is blank, returns nil."
  [value]
  (if (and value (not (str/blank? value)))
    (try
      (Float/parseFloat value)
      (catch NumberFormatException e
        (failure 406 "Invalid number format for" value)))
    nil))

;; Taken from clojure.contrib.ns-utils 1.2.0
(defn immigrate
  "Create a public var in this namespace for each public var in the
  namespaces named by ns-names. The created vars have the same name, root
  binding, and metadata as the original except that their :ns metadata
  value is this namespace."
  [& ns-names]
  (doseq [ns ns-names]
    (require ns)
    (doseq [[sym var] (ns-publics ns)]
      (let [sym (with-meta sym (assoc (meta var) :ns *ns*))]
        (if (.hasRoot var)
          (intern *ns* sym (.getRawRoot var))
          (intern *ns* sym))))))

(defn valid-slug?
  "Ensure the slug doesn't have a colon, ampersand, slash, asterisk, or
  semi-colon in it."
  [slug]
  (if-not (nil? (re-find #"[:&/\*;]" slug))
    (failure 406 "The given slug, '" slug "', is invalid.  A slug may not contain colons, ampersands, slashes, asterisks, or semi-colons.")))

(defn to-slug
  "Convert the given string to an SEO-compatible slug.  Lowercases the string,
  and replaces all spaces or underscores with dashes.  Any other
  non-alphanumeric character is removed.

  This can be useful for local testing, but it's better to call the Singularity
  server for a slug, so the rules are consistent."
  [string]
  (if (nil? string)
    nil
    (->
     (str/lower-case string)
     (str/trim)
     (str/replace #"[^\w-_\s:/]+" "")
     (str/replace #"[_\s:/]+" "-"))))
