(ns singularity.request

  "Support for handling HTTP requests in Singularity."

  (:require
    [clojure.string :as str]
    [clojure.data.json :as json]
    [com.twinql.clojure.http :as http]
    [tandem.settings :as settings])
  (:import
    singularity.SingularityException))

(def SINGULARITY (settings/for :singularity.url))
(def ^:dynamic *credentials* nil)

(defn- encode-path
  [s]
  (let [s (if (keyword? s) (name s) (str s))]
    (str/replace (java.net.URLEncoder/encode s "UTF-8") "+" "%20")))

(defn to
  "Create a full URL to the given path resource.  For example,

    (to \"p\" \"some-slug\" \"m\")

  would return a URL that looks like:

    http://localhost:8080/p/some-slug/m
  "
  [& segments]
  (str SINGULARITY "/" (str/join "/" (map encode-path segments))))

(defn- make-request
  [http-func url params format]
  (let [params (if *credentials* (merge {:user_credentials *credentials*} params) params)
        format (or format :json)]
    (let [response (http-func url :query params :as :string)
          {status :code result :content} response
          result (if (= format :json) (try
                                        (json/read-json result)
                                        (catch Throwable t
                                          result)) result)]
      (if (< status 400)
        result
        (if (map? result)
          (throw (SingularityException. status (:error result) (:duplicate result)))
          (throw (SingularityException. status (str "Request Failed: " result))))))))

(defn GET [url & [params format]] (make-request http/get url params format))
(defn POST [url & [params format]] (make-request http/post url params format))
(defn PUT [url & [params format]] (make-request http/put url params format))
(defn DELETE [url & [params format]] (make-request http/delete url params format))
