(ns singularity.rabbit

  "Some wrappers for RabbitMQ, to make it Clojurific."

  (:use singularity.support)
  (:require [clojure.data.json :as json]
            [clojure.tools.logging :as log]
            [singularity.settings :as settings])
  (:import [com.rabbitmq.client ConnectionFactory Channel QueueingConsumer AMQP$BasicProperties$Builder]
           [java.util Calendar TimeZone]
           [brilliantarc.singularity SingularityException]))

(def ^:dynamic *channel* nil)
(def ^:dynamic *exchange* nil)
(def ^:dynamic *queue* nil)
(def ^:dynamic *routing-key* nil)

(def settings (:rabbit config/settings))

(defn exchange-settings
  "Get the settings (exchange, queue) for an individual exchange."
  [name]
  (get settings name))

(defn connect
  "Open a connection to RabbitMQ.  Expects options for :host, :port, :user,
  and :password."
  [options]
  (let [username (or (:user options) ConnectionFactory/DEFAULT_USER)
        password (or (:password options) ConnectionFactory/DEFAULT_PASS)
        host (or (:host options) ConnectionFactory/DEFAULT_HOST)
        port (or (:port options) ConnectionFactory/DEFAULT_AMQP_PORT)]
    (try
      (.newConnection (doto (ConnectionFactory.)
                        (.setUsername username)
                        (.setPassword password)
                        (.setHost host)
                        (.setPort port)))
      (catch java.net.ConnectException e
        (failure 500 "There is either a network problem or the message queue is offline.")))))

(defn channel
  "Open a channel on a RabbitMQ connection."
  [connection]
  (.createChannel connection))

(defmacro with-channel
  "This is primarily used internally.  Use with-exchange instead."
  [& body]
  `(if *channel*
     (do ~@body)
     (with-open [connection# (connect (:mq settings))
                 channel# (channel connection#)]
       (binding [*channel* channel#]
         ~@body))))

(defn declare-exchange
  "Create or reference an exchange.  Type may be 'direct', 'topic', or 'fanout',
  or use one of the methods like direct-exchange.  If durable, messages will be
  sync'd to disk to survive a reboot of the RabbitMQ server."
  ([channel name type durable] (.exchangeDeclare channel name type durable))
  ([name type durable] (declare-exchange *channel* name type durable)))

(defn direct-exchange
  ([channel name durable] (declare-exchange channel name "direct" durable))
  ([name durable] (declare-exchange *channel* name "direct" durable)))

(defn topic-exchange
  ([channel name durable] (declare-exchange channel name "topic" durable))
  ([name durable] (declare-exchange *channel* name "topic" durable)))

(defn fanout-exchange
  ([channel name durable] (declare-exchange channel name "fanout" durable))
  ([name durable] (declare-exchange *channel* name "fanout" durable)))

(defn declare-queue
  "Create or reference a queue.

    name:       the name of the queue
    durable:    true, the queue will survive a server restart
    exclusive:  true, restricted to this connection
    autoDelete: true, server will delete it when no longer in use

  "
  ([channel name durable exclusive auto-delete] (.queueDeclare channel name durable exclusive auto-delete nil))
  ([channel name] (declare-queue channel name true false false))
  ([name durable exclusive auto-delete] (declare-queue *channel* name durable exclusive auto-delete))
  ([name] (declare-queue *channel* name)))

(defn bind-queue
  "Bind an exchange to a queue."
  ([channel queue-name exchange-name routing-key] (.queueBind channel queue-name exchange-name routing-key))
  ([queue-name exchange-name routing-key] (bind-queue *channel* queue-name exchange-name routing-key)))

(defn qos
  "Determine how many messages will be delivered to a channel.  Typically set
  to 1 for round-robin load balancing."
  ([channel prefetch-count] (.basicQos channel prefetch-count))
  ([prefetch-count] (qos *channel* prefetch-count)))

(defn now
  "Return right now in GMT/UTC."
  []
  (.getTime (Calendar/getInstance (TimeZone/getTimeZone "GMT"))))

(defn uuid
  "Return a globally unique ID."
  []
  (str (java.util.UUID/randomUUID)))

(defn basic-properties
  "Configures our message for JSON.  If you are waiting for a reply, submit the
  queue name and a correlation ID will be generated and set, along with the
  reply queue."
  [& [reply-to correlation-id]]
  (let [builder (->
    (AMQP$BasicProperties$Builder.)
    (.contentType "application/json")
    (.timestamp (now)))]

    ;; Do we need to include a reply?
    (when reply-to (.replyTo builder reply-to))
    (when correlation-id (.correlationId builder correlation-id))

    (.build builder)))

(defn initialize-exchange
  "Setup a queue and exchange."
  [name]
  (let [{{queue :queue exchange :exchange routing-key :routing-key} name} settings]
    (with-channel
      (qos 1)
      (declare-queue queue)
      (direct-exchange exchange true)
      (bind-queue queue exchange routing-key))))

(defmacro with-exchange
  "Used to wrap publish requests."
  [exchange-name & body]
  `(let [{{exchange# :exchange routing-key# :routing-key} ~exchange-name} settings]
     (if exchange#
       (with-channel
         (binding [*exchange* exchange# *routing-key* routing-key#]
           ~@body))
       (throw (IllegalArgumentException. (str "Please indicate a exchange defined in terra.json:" ~name))))))

(defmacro with-queue
  "Used to wrap consume requests."
  [queue-name & body]
  `(let [{{queue# :queue} ~queue-name} settings]
     (if queue#
       (with-channel
         (binding [*queue* queue#]
           ~@body))
       (throw (IllegalArgumentException. (str "Please indicate a queue defined in terra.json:" ~name))))))

(declare publish)

(defn create-consumer
  "Create a message consumer.  Whenever a message is received, it is passed
  through the function with the routing key, delivery tag, and the message as
  a keywordized map, parsed from JSON.  Expects you to acknowledge each receipt,
  i.e. autoAck is false.  Returns the generated consumer tag.

  For safety, the function will NOT receive the channel.  Accessing certain
  functions on a channel can block the thread and lock the consumer.  So we
  just don't allow access to the channel itself.

  For example:

    (create-consumer some-channel some-queue
      (fn [routing-key delivery-tag message]
      ...do something to the message; raise an exception on failure...))

  Note that this will attach to the current thread.  Better to wrap this call
  in a Future or another threading process."
  ([channel queue-name func]
    (let [consumer (QueueingConsumer. channel)]
      (.basicConsume channel queue-name false consumer)
      (loop [delivery (.nextDelivery consumer)]
        (let [envelope (.getEnvelope delivery)
              routing-key (.getRoutingKey envelope)
              delivery-tag (.getDeliveryTag envelope)
              properties (.getProperties delivery)
              content-type (.getContentType properties)]

          ;; We only accept JSON messages
          (if (= content-type "application/json")
            (let [message (json/read-json (String. (.getBody delivery) "UTF-8"))]
              (try

                ;; Run our handler function
                (let [value (func routing-key delivery-tag message)
                      correlation-id (.getCorrelationId properties)
                      reply-to (.getReplyTo properties)]

                  ;; Do we need to send a reply?
                  (if (and value correlation-id)
                    (let [message-json (json/json-str value)
                          message-bytes (.getBytes message-json "UTF-8")]
                      (log/debug "Sending RPC reply:" message)
                      (.basicPublish channel "" reply-to (basic-properties nil correlation-id) message-bytes))))

                ;; Acknowledge the message if the function completes without error
                (.basicAck channel delivery-tag false)

                (log/debug "Handled message" message)

                (catch Throwable t
                  (log/error t "Error with message from RabbitMQ:" (.getMessage t))
                  (log/error "Queue:" *queue* ", routing key:" routing-key)
                  (log/error "Message was:" (String. (.getBody delivery)))
                  (.basicReject channel delivery-tag false)

                  ;; If the requester is waiting for a reply, let's oblige
                  (let [correlation-id (.getCorrelationId properties)
                        reply-to (.getReplyTo properties)]
                    (if correlation-id
                      (let [message-json (json/json-str {:error (.getMessage t)})
                            message-bytes (.getBytes message-json "UTF-8")]
                        (.basicPublish channel "" reply-to (basic-properties nil correlation-id) message-bytes)))))))


            ;; Reject the message if it's not JSON
            (.basicReject channel delivery-tag false)))

        ;; TODO:  Currently an endless loop...
        (recur (.nextDelivery consumer)))))

  ([queue-name func] (create-consumer *channel* queue-name func)))

(defn consume
  "Creates a consumer of RabbitMQ messages and wraps it in a future, so it is
  handled on another thread and doesn't block.  Indicate the queue settings
  to use from terra.json by name.  Since we mix exchange and queue settings
  together, you can really use any settings to initialize the queue and setup
  a consumer, and it's ok to call this repeatedly (just to be safe).

  This call returns a future.  If you request its return value, your current
  thread will block and wait for it to return (which it never will).

  You may specify how many consumers to configure for the given queue/function
  combination.  If your handlers can work in a multi-threaded, parallel fashion,
  this can greatly improve performance.  Sort of like a built-in Hadoop."
  [queue-name func]
  (initialize-exchange queue-name)
  (let [{{consumers :consumers :or {consumers 1}} queue-name} settings]
    (log/debug "Creating" consumers "consumers for" queue-name)
    (dotimes [pass consumers]
      (future (with-queue queue-name (create-consumer *queue* func))))))

(defn consume?
  "Like the consume function, but if the queue doesn't exist, it just won't
  create the consumer.  Useful for dev or test queues."
  [queue-name func]
  (if (contains? settings queue-name)
    (consume queue-name func)))

(defn publish
  "Publish a message to an exchange with the given routing key.  Wrap in a
  with-exchange macro, to identify where to publish the message.

  Currently, because we deal in JSON, the message always routes in JSON.  This
  may change later, but unlikely, and thus this is not a general-purpose wrapper
  for RabbitMQ, per se.  Also, the message is expected to be a map, which will
  be turned into a JSON object, converted to bytes, and posted to RabbitMQ.

  If response is set to true (defaults to false), the publish request will
  setup an RPC queue and return a future that is listening for the response.
  The message handler is expected to return a result, which will be passed back
  to the RPC queue and returned by dereferencing the future."
  ([routing-key message response]
    (let [message-json (json/json-str message)
          message-bytes (.getBytes message-json "UTF-8")
          routing-key (name routing-key)]
      (log/debug (str "Publishing message \"" message-json "\" to " routing-key))

      ;; Does the user expect a response?  If so, let's setup a reply queue
      (if-not response
        ;; Response not needed...just a standard fire-and-forget
        (.basicPublish *channel* *exchange* routing-key true false (basic-properties) message-bytes)

        ;; Response requested:  we need a temporary queue to listen for the
        ;; reply, and return a future to the user with the reply in it.
        (let [reply-queue-name (.getQueue (.queueDeclare *channel*))
              consumer (QueueingConsumer. *channel*)
              correlation-id (uuid)]

          ;; Setup the RPC queue to handle one message...one ping only, Vasili
          (.basicConsume *channel* reply-queue-name true consumer)

          ;; Publish our message
          (let [properties
                (-> (AMQP$BasicProperties$Builder.)
                  (.contentType "application/json")
                  (.timestamp (now))
                  (.correlationId correlation-id)
                  (.replyTo reply-queue-name)
                  (.build))]
            (.basicPublish *channel* *exchange* routing-key true false properties message-bytes))

          ;; Block and wait for the response
          (loop [delivery (.nextDelivery consumer)]
            (let [properties (.getProperties delivery)
                  sent-correlation-id (.getCorrelationId properties)]
              (if (= sent-correlation-id correlation-id)
                (let [response (json/read-json (String. (.getBody delivery) "UTF-8"))]
                  (if (contains? response :error )
                    (failure 500 (:error response))
                    response))
                (recur (.nextDelivery consumer)))))))))

  ([message response] (publish *routing-key* message response))
  ([message] (publish *routing-key* message nil)))

(defn publish!
  "Like calling (publish message true).  Just a quick shortcut."
  ([routing-key message] (publish routing-key message true))
  ([message] (publish *routing-key* message true)))


