(ns bridg.sqs.core
  "Lightweight wrapper around SQS functionality to abstract away details for
  high-level SQS consumers."
  (:require
    [amazonica.aws.sqs :as sqs]
    [cheshire.core :as json]
    [schema.core :as s]
    [taoensso.timbre :refer [info warn]])
  (:import
    com.amazonaws.auth.DefaultAWSCredentialsProviderChain
    com.amazonaws.services.sqs.model.MessageNotInflightException
    com.amazonaws.services.sqs.model.QueueDoesNotExistException
    com.amazonaws.services.sqs.model.ReceiptHandleIsInvalidException))

(defn produce!
  "Produces a message to SQS using the send-message API.

  MessageGroupId is similar to a partition. Only one message will be retrieved
  for a given message group id at a time.

  Messages must be JSON (only XML, JSON and raw text are supported by SQS.)"
  [queue-url message-group-id message-body]
  (sqs/send-message
    :queue-url queue-url
    :delay-seconds 0
    :message-body (json/generate-string message-body)
    :message-group-id message-group-id))

(defn consume!
  "Consumes a message from SQS using the receive-message API."
  [queue-url]
  (let [attribute-names ["All"]  ; list of attributes to return in each message
        delete false             ; unknown, auto-delete message on retrieval?
        max-number-of-messages 1 ; valid values are 1-10
        ;; TODO: Make visibility-timeout configurable.
        visibility-timeout 30    ; duration (in seconds) that recieved messages are hidden
        ;; TODO: Make wait-time-seconds configurable.
        wait-time-seconds 20]    ; duration (in seconds) for which the call waits for a message to arrive
    (sqs/receive-message
      (DefaultAWSCredentialsProviderChain.)
      :queue-url queue-url
      :wait-time-seconds wait-time-seconds
      :max-number-of-messages max-number-of-messages
      :delete delete
      :attribute-names ["All"])))

(defn change-message-visibility
  "Changes the visibility timeout of a specified message in a queue to a new
  value. The maximum allowed timeout value is 12 hours. Thus, you can't extend
  the timeout of a message in an existing queue to more than a total visibility
  timeout of 12 hours. For more information, see Visibility Timeout in the
  Amazon Simple Queue Service Developer Guide.

  http://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibility.html"
  [queue-url receipt-handle visibility-timeout]
  (sqs/change-message-visibility queue-url receipt-handle visibility-timeout))

(defn delete!
  [queue-url receipt-handle]
  (sqs/delete-message queue-url receipt-handle))

;;
(defn first-message-body
  "Returns the body of the first message returned from SQS."
  [result]
  (-> result :messages first :body))

(defn message-bodies
  "Returns sequence of all bodies from the SQS result."
  [result]
  (map :body (:messages result)))
