(ns bridg.sqs.component.sqs
  "Stateful SQS consumer."
  (:require
    [bridg.sqs.core :as sqs]
    [cheshire.core :as json]
    [com.stuartsierra.component :as component]
    [schema.core :as s]
    [taoensso.timbre :refer [debug info warn]])
  (:import
    com.amazonaws.auth.DefaultAWSCredentialsProviderChain
    com.amazonaws.services.sqs.model.MessageNotInflightException
    com.amazonaws.services.sqs.model.QueueDoesNotExistException
    com.amazonaws.services.sqs.model.ReceiptHandleIsInvalidException))


(defprotocol ActivatedConsumer
  "Protocol for finding the active consumer from a configured list."
  (active-consumer-key
    [this]
    "Gets the key of the currently active consumer.")
  (active-consumer-config
    [this]
    "Gets the config of he currently active consumer.")
  (active-consumer-component
    [this]
    "Finds the active consumer component in the system. Assumes that the 
    component has the same name as the component key."))

(defprotocol Subscriber
  (subscribe!
    [this]
    "Subscribes to a queue and polls for messages to dispatch to handle-fn."))

(defn default-handler [component queue-url sqs-result complete-fn extend-fn]
  (complete-fn))

(defn sqs-response-wrapper [component queue-url sqs-result]
  (if (-> sqs-result :messages not-empty)
    (doseq [message (-> sqs-result :messages)
            :let [{:keys [receipt-handle]} message
                  complete-fn #(sqs/delete! queue-url receipt-handle)
                  extend-fn (partial sqs/change-message-visibility queue-url receipt-handle)
                  handler (-> component active-consumer-config :handler)]]
      (debug :msg "SQS queued message found."
             :event "sqs.consumer.poll.message-found"
             :queue-url queue-url
             :sqs-result sqs-result)
      ;; Call external handler.
      (handler (active-consumer-component component) queue-url message
               complete-fn extend-fn)
      (info :msg "Completed SQS message by deletion."
            :event "sqs.consumer.message.completed"
            :receipt-handle receipt-handle))
    (debug :msg "SQS consumer found no queued messages."
           :event "sqs.consumer.poll.no-message-found"
           :queue-url queue-url)))

(defrecord SQS [config]
  ActivatedConsumer
  (active-consumer-key [this]
    (-> this :config :active-consumer))
  (active-consumer-config [this]
    (get (-> this :config :consumers) (active-consumer-key this)))
  (active-consumer-component [this]
    (get this (active-consumer-key this)))

  Subscriber
  ;; TODO: Error handling.
  (subscribe! [this]
    (when (-> this :config :enabled?)
      (let [{:keys [queue-url handler error-fn]}
            (active-consumer-config this)]
        (reset! (:subscribed this) true)
        (info :msg "SQS consumer subscribed to queue."
              :event "sqs.consumer.subscribed"
              :queue-url queue-url)
        (while (not @(:shutdown this))
          (let [sqs-result (sqs/consume! queue-url)]
            (sqs-response-wrapper this queue-url sqs-result)))
        (reset! (:subscribed this) false)
        (info :msg "SQS consumer unsubscribed to queue."
              :event "sqs.consumer.unsubscribed"
              :queue-url queue-url))))

  component/Lifecycle
  (start [this]
    (if (:enabled? config)
      (do
        (info :msg "Started SQS consumer component."
              :event "sqs.consumer.started")
        (assoc this
               :shutdown (atom false)
               :subscribed (atom false)))
      (do
        (warn :msg (str "SQS consumer component disabled! "
                        "Messages will not be consumed.")
              :event "sqs.consumer.disabled")
        this)))

  (stop [{:keys [consumer options] :as this}]
    (reset! (:shutdown this) true)
    (while @(:subscribed this) (Thread/sleep 100))
    (info :msg "Stopped SQS consumer component."
          :event "sqs.consumer.stopped")
    (dissoc this :shutdown :subscribed)))

(defn sqs [config]
  (->SQS config))
