(ns bridg.logfmt
  ^{:doc "Provides logfmt formatter for Timbre.
         Formatters and wrappers around Taoensso Timbre with a customizations
         for strict event attribute sorting and logfmt formatting."}
  (:require
    [com.stuartsierra.component :as component]
    [cheshire.core :as json]
    [clojure.string :as string]
    [taoensso.timbre :as timbre]))

(def
  ^{:dynamic true
    :doc "Thread-specific, dynamic, logging context"}
  *context* nil)

(defn context
  "Returns the current logging context."
  []
  *context*)

(defn ^String string-contains?
  "Returns true if s contains the substring."
  [#^String s substring]
  (.contains s substring))

(defn escape-quotes [s]
  "Replace double quote chars with escaped quotes."
  (string/replace s #"\"" "\""))

(defn safestring [v]
  "Formats a string value for use in key=value pairs."
  (if (string-contains? v " ")
    (str "\"" (escape-quotes v) "\"")
    (escape-quotes v)))

(defprotocol Logfmt
  (format-value [v] "Formats a type into a logfmt friendly string."))

(extend-protocol Logfmt
  java.lang.String
  (format-value [v] (safestring v))
  clojure.lang.Keyword
  (format-value [v] v)
  clojure.lang.PersistentArrayMap
  (format-value [v] (json/generate-string v))
  nil
  (format-value [v] (json/generate-string v))
  java.lang.Object
  (format-value [v] (safestring (str v))))

(defn ^String msg
  "Takes a series of k/v pairs and turns them into a logfmt formatted string
  while retaining the order of the pairs."
  [& pairs]
  (string/join " "
    (map (fn [[k v]] (str (name k) "=" (format-value v)))
        (partition 2 pairs))))

(defn map->logfmt
  "Formats a map into a string of key=value pairs."
  [m]
  (string/join " " (map (fn [[k v]]
                          (str (name k) "=" (format-value v)))
                        m)))

(defn logfmt-formatter
  "A contextual logfmt logging appender."
  [{:keys [vargs level timestamp_ hostname_ ?line ?ns-str context] :as ev}
   & appender-fmt-output-opts]
  (let [pairs (concat (vec context) vargs)
        kv (apply msg pairs)]
    (format "level=%s %s logger.ns=%s:%s ts=%s"
            (name level) kv ?ns-str ?line @timestamp_)))

(def timbre-config-overrides
  "Timbre configuration overrides."
  {:output-fn logfmt-formatter
   :timestamp-opts {:pattern "yyyy-MM-dd'T'HH:mm:ss.SSSZ" :timezone :utc}})

(def timbre-config
  "Full Timbre example configuration with customized overrides."
  (timbre/merge-config! timbre-config-overrides))

(defmacro with-context
  "Binds the context to the dynamic log context and evaluates body. Differs from
  the standard Timbre context in that it's stored as a vector to preserve order
  and continually appends the context via `concat` rather than completely
  replacing it.

  Example:

    #_=> (with-context [:key1 :foo
                        :key2 :bar]
           (info :msg \"Something happened.\"))

    level=info key1=:foo key2=:bar msg=\"Something happened.\"
  "
  [context & body]
  `(binding [timbre/*context* (concat timbre/*context* ~context)]
     ~@body))
