(ns bridg.test-helper
  "Common functions to help with testing."
  (:require
    [cheshire.core :as cheshire]
    [clj-time.core :refer [after?]]
    [clj-time.jdbc]
    [clojure.edn :as edn]
    [clojure.java.io :as io]
    [clojure.java.jdbc :as sql]
    [clojure.test :refer :all]
    [com.stuartsierra.component :as component]
    [compojure.api.sweet :refer [api]]
    [duct.component.ragtime :refer [migrate]]
    [duct.util.system :refer [load-system read-config]]
    [{{namespace}}.system :as system]
    [bridg.fixtures :as fixtures])
  (:import
    [java.util Date]
    [org.joda.time DateTime]))

(declare ^:dynamic *system*)
(declare ^:dynamic *db*)

(defmacro pending
  "Defines a pending test with no arguments. A pending message will be printed
  when test output is enabled, but pending tests aren't reported in final test
  totals."
  [name & body]
  (let [message (str "\n" name " is pending!!")]
    `(testing ~name (println ~message))))

(defn test-system
  "Loads a new system from edn."
  []
  (load-system (keep io/resource ["bridg/audience_export_group/system.edn"
                                  "test.edn"
                                  "local-test.edn"])))

(defn with-test-system [f]
  "Binds a test system to the *system* dynamic variable, which may be used
  within tests."
  (binding [*system* (component/start
                       (system/new-system
                          ["{{dirs}}/system.edn"
                           "test.edn"
                           "local-test.edn"]))]
    (try
      (f)
      (finally
        (component/stop *system*)))))

(defn with-component
  "Returns a fixture function to bind a component to the *db* dynamic
  variable as found from korks in the *system*. Assumes that *system* is
  already defined, so must be used after 'with-test-system!"
  [korks]
  (let [korks (if (sequential? korks) korks [korks])]
    (fn [f]
      (let [db (get-in *system* korks)]
        (binding [*db* db]
          (f))))))

(defn with-components
  "Returns a fixture function that binds system components to dynamic vars. The
  bindings are pairs of a namespaced quoted symbol (ex #'*db*) and a korks (key
  or keys) to find the component in the *system*. Assumes that *system* is
  already defined, so must be used after 'with-test-system!

  Example Usage

  Following example binds `(:brand *system*)` to var `*brand*` and
  `(:account system)` to var `*account*`:

      (use-fixtures
        :each (join-fixtures [helper/with-test-system
                              helper/with-migrations
                              (helper/with-components [#'*brand* :brand
                                                       #'*account* :account])
                              (helper/with-clean-db clean-tables)]))
  "
  [bindings]
  (fn [f]
    (let [pairs (partition 2 bindings)
          formatter (fn [[b korks]]
                      (let [korks (if (sequential? korks) korks [korks])
                            component (get-in *system* korks)]
                        {b component}))
          fpairs (map formatter pairs)
          fmap (into {} fpairs)]
      (with-bindings fmap (f)))))

(defn with-rollback-components
  "Expects a list of dynamic vars to top level system component keywords. Each top-level system
  component is also expected to contain a nested [:db :spec] structure available. Each component
  will be rebounded with a database connection that will rollback.

  Example Usage

  Following example binds `(:brand *system*)` to var `*brand*` and
  `(:account system)` to var `*account*`, all DB actions will rollback:

      (use-fixtures
        :each (join-fixtures [helper/with-test-system
                              (helper/with-rollback-components [#'*brand* :brand
                                                                #'*account* :account])]))
  "
  [bindings]
  (fn [f]
    (sql/with-db-transaction [db (get-in *system* [:db :spec])]
      (sql/db-set-rollback-only! db)
      (let [pairs (partition 2 bindings)
            formatter (fn [[b k]]
                        {b (assoc-in (k *system*) [:db :spec] db)})
            fpairs (map formatter pairs)
            fmap (into {} fpairs)]
        (with-bindings fmap (f))))))

(defn with-migrations
  "Ensure database migrations are up-to-date."
  [f]
  (migrate (:ragtime *system*))
  (f))

(defn with-clean-db
  "Runs cleaning-fn prior to test to ensure clean db state."
  [cleaning-fn]
  (fn [f]
    (cleaning-fn)
    (f)))

(defn load-seeds
  "Loads seeds by name from edn."
  [path]
  (let [fullpath (str "{{dirs}}/seeds/" path ".edn")
        content  (slurp (io/resource fullpath))]
    (edn/read-string content)))

(defn with-db-fixtures
  "Loads fixture data. Assumes that *system* is already defined, so must be used
  after 'with-test-system!"
  [f]
  (fixtures/load! (:db *system*) (load-seeds "fixtures"))
  (f))

(defn api-handler
  "Applies test routes to a compojure-api api."
  [& routes]
  (apply api routes))

(defn now [] (new java.util.Date))

(defn to-json-types
  "Converts to lossy JSON and back again to simulate a datastructure response
  expected from the API."
  [x]
  (-> x
      cheshire/generate-string
      cheshire/parse-string))

(defn parse-body [body]
  (when body
    (-> body
        slurp
        (cheshire/parse-string))))
