(ns kafkakit.avro
  "Convert Clojure data structures to Avro-compatible Java classes (not Avro
  binary format!) back and forth in order to be able to use Schema Registry's
  serializers with Kafka and Clojure.

  NOTE: this doesn't include processing on logical type duration. There is no
  processing necessary because it is just a byte encoding, not a specific
  language type.

  Source: https://github.com/konukhov/kfk-avro-bridge

  References:

    * https://github.com/confluentinc/schema-registry/blob/master/avro-serializer/src/main/java/io/confluent/kafka/serializers/KafkaAvroSerializer.java
  "
  (:refer-clojure :exclude [bytes?])
  (:require
    [clj-time.coerce :as c])
  (:import
    [java.nio ByteBuffer]
    [org.apache.avro.generic GenericData]
    [org.apache.avro.generic GenericData$Array]
    [org.apache.avro.generic GenericData$EnumSymbol]
    [org.apache.avro.generic GenericData$Fixed]
    [org.apache.avro.generic GenericData$Record]
    [org.apache.avro.util Utf8]
    [org.apache.avro Schema Schema$Parser Schema$Type
     Conversions$DecimalConversion LogicalType LogicalTypes]
    [org.apache.avro.data TimeConversions$DateConversion
     TimeConversions$TimeConversion TimeConversions$TimestampConversion]
    [java.math BigDecimal BigInteger]
    [org.joda.time LocalDate]))

(defn- bytes?
  "Compatibility with Clojure < 1.9"
  [x]
  (if (nil? x)
    false
    (-> x class .getComponentType (= Byte/TYPE))))

(defn- throw-invalid-type
  [^Schema schema obj]
  (throw (Exception. (format "Value `%s` cannot be cast to `%s` schema"
                             (str obj)
                             (.toString schema)))))

(defn parse-schema
    "A little helper for parsing schemas"
  [json]
  (.parse (Schema$Parser.) json))

(defn logical-name [schema]
  (try ;; Some types don't have the getLogicalType method
    (.getName (.getLogicalType schema))
    (catch Exception e nil)))

(defn ->java
  "Converts a Clojure data structure to an Avro-compatible
     Java object. Avro `Schema` must be provided."
  [schema obj]
  (condp = (and (instance? Schema schema) (.getType schema))
    Schema$Type/NULL
    (if (nil? obj)
      nil
      (throw-invalid-type schema obj))

    Schema$Type/INT
    (condp = (logical-name schema)
      "date"
      (.toInt (TimeConversions$DateConversion.) obj schema
              (.getLogicalType schema))

      "time-millis"
      (.toInt (TimeConversions$TimeConversion.) obj schema
              (.getLogicalType schema))

      (if (and (integer? obj) (<= Integer/MIN_VALUE obj Integer/MAX_VALUE))
        (int obj)
        (throw-invalid-type schema obj)))

    Schema$Type/LONG
    (condp = (logical-name schema)
      ;; NOTE: Joda Time does not support microsecond accurate time.
      ;; The official Avro Java libraries use Joda Time and do not support
      ;; a round trip conversion. Once Apache moves to java.time, this can
      ;; be updated. I didn't want to mix the LocalTime (Java vs. java.time)
      ;; types between logical Avro types.
      "time-micros"
      (throw (Exception. "Time-micros is unsupported."))

      "timestamp-millis"
      (.toLong (TimeConversions$TimestampConversion.) obj schema
               (.getLogicalType schema))

      "timestamp-micros"
      (throw (Exception. "Time-micros is unsupported."))

      (if (and (integer? obj) (<= Long/MIN_VALUE obj Long/MAX_VALUE))
        (long obj)
        (throw-invalid-type schema obj)))

    Schema$Type/FLOAT
    (if (float? obj)
      (float obj)
      (throw-invalid-type schema obj))

    Schema$Type/DOUBLE
    (if (float? obj)
      (double obj)
      (throw-invalid-type schema obj))

    Schema$Type/BOOLEAN
    (if (instance? Boolean obj) ;; boolean? added only in 1.9 :(
      obj
      (throw-invalid-type schema obj))

    Schema$Type/STRING
    (cond
      (string? obj) obj
      (instance? java.util.UUID obj) (str obj)
      :default (throw-invalid-type schema obj))

    Schema$Type/BYTES
    (cond
      (= "decimal" (logical-name schema))
      (if (instance? java.math.BigDecimal obj)
        (let [logical-type (.getLogicalType schema)]
          (.toBytes (Conversions$DecimalConversion.)
                    obj schema (LogicalTypes/decimal
                                (.getPrecision logical-type)
                                (or (.getScale logical-type) 0))))
        (throw (Exception.
                (str "Logical type decimal must be passed a "
                     "java.math.BigDecimal."))))

      :else
      (if (bytes? obj)
        (doto (ByteBuffer/allocate (count obj))
          (.put obj)
          (.position 0))
        (throw-invalid-type schema obj)))


    Schema$Type/ARRAY
    (let [f (partial ->java (.getElementType schema))]
      (GenericData$Array. schema (map f obj)))

    Schema$Type/FIXED
    (cond
      (= "decimal" (logical-name schema))
      (if (instance? java.math.BigDecimal obj)
        (let [logical-type (.getLogicalType schema)]
          (.toFixed (Conversions$DecimalConversion.)
                    obj schema (LogicalTypes/decimal
                                (.getPrecision logical-type)
                                (or (.getScale logical-type) 0))))
        (throw (Exception.
                (str "Logical type decimal must be passed a "
                     "java.math.BigDecimal."))))

      :else
      (if (and (bytes? obj) (= (count obj) (.getFixedSize schema)))
        (GenericData$Fixed. schema obj)
        (throw-invalid-type schema obj)))

    Schema$Type/ENUM
    (let [enum-name (name obj)
          ns (.getNamespace schema)
          enums (into #{} (.getEnumSymbols schema))]
      (if (or (string? obj)
              (nil? (namespace obj))
              (= ns (namespace obj)))
        (if (contains? enums enum-name)
          (GenericData$EnumSymbol. schema enum-name)
          (throw-invalid-type schema enum-name))
        (throw (Exception.
                (str "Namespace of passed enum, "
                     (namespace obj)
                     ", does not match namespace of schema, " ns ".")))))

    Schema$Type/MAP
    (zipmap (map name (keys obj))
            (map (partial ->java (.getValueType schema))
                 (vals obj)))

    Schema$Type/UNION
    (let [[val matched]
          (reduce (fn [_ schema]
                    (try
                      (reduced [(->java schema obj) true])
                      (catch Exception _
                        [nil false])))
                  [nil false]
                  (.getTypes schema))]
      (if matched
        val
        (throw-invalid-type schema obj)))

    Schema$Type/RECORD
    (do
      (reduce-kv
       (fn [record k v]
         (let [k (name k)
               s (some-> (.getField schema k)
                         (as-> f (.schema f)))]
           (doto record
             (.put k (->java (or s k) v)))))
       (GenericData$Record. schema)
       obj))

    (throw (Exception. (format "Field `%s` is not in schema" schema)))))

(defn ->clj
  "Parses deserialized Avro object into a Clojure data structure."
  [schema msg]
  (condp = (and (instance? Schema schema) (.getType schema))
    Schema$Type/NULL
    (if (nil? msg)
      nil
      (throw-invalid-type schema msg))

    Schema$Type/STRING
    (if (string? msg)
      msg
      (throw-invalid-type schema msg))

    Schema$Type/FLOAT
    (if (float? msg)
      (float msg)
      (throw-invalid-type schema msg))

    Schema$Type/DOUBLE
    (if (float? msg)
      (double msg)
      (throw-invalid-type schema msg))

    Schema$Type/BOOLEAN
    (if (instance? Boolean msg) ;; boolean? added only in 1.9 :(
      msg
      (throw-invalid-type schema msg))

    Schema$Type/ARRAY
    (let [f (partial ->clj (.getElementType schema))]
      (map f msg))

    Schema$Type/FIXED
    (cond
      (= "decimal" (logical-name schema))
      (let [decimal-conversion (Conversions$DecimalConversion.)
            logical-type (.getLogicalType schema)
            precision (.getPrecision logical-type)
            scale (.getScale logical-type)]
        (.fromFixed decimal-conversion msg schema logical-type))

      :else (.bytes msg))

    Schema$Type/LONG
    (condp = (logical-name schema)
      ;; See note in ->java on micros not being supported.
      "time-micros"
      (throw (Exception. "Time-micros is unsupported."))

      "timestamp-millis"
      (.fromLong (TimeConversions$TimestampConversion.) msg schema
                 (.getLogicalType schema))

      "timestamp-micros"
      (throw (Exception. "Time-micros is unsupported."))

      (if (and (integer? msg) (<= Long/MIN_VALUE msg Long/MAX_VALUE))
        (long msg)
        (throw-invalid-type schema msg)))

    Schema$Type/INT
    (condp = (logical-name schema)
      "date"
      (.fromInt (TimeConversions$DateConversion.) msg schema
                (.getLogicalType schema))

      "time-millis"
      (.fromInt (TimeConversions$TimeConversion.) msg schema
                (.getLogicalType schema))

      (if (and (integer? msg) (<= Integer/MIN_VALUE msg Integer/MAX_VALUE))
        (int msg)
        (throw-invalid-type schema msg)))


    Schema$Type/ENUM
    (let [ns (.getNamespace schema)
          enum msg
          enums (into #{} (.getEnumSymbols schema))]
      (if (contains? enums (str enum))
        (if ns
          (keyword ns (str enum))
          (keyword (str enum)))
        (throw-invalid-type schema msg)))

    Schema$Type/UNION
    (let [[val matched]
          (reduce (fn [_ schema]
                    (try
                      (reduced [(->clj schema msg) true])
                      (catch Exception _
                        [nil false])))
                  [nil false]
                  (.getTypes schema))]
      (if matched
        val
        (throw-invalid-type schema msg)))

    Schema$Type/BYTES
    (cond
      (= "decimal" (logical-name schema))
      (let [decimal-conversion (Conversions$DecimalConversion.)
            logical-type (.getLogicalType schema)
            precision (.getPrecision logical-type)
            scale (.getScale logical-type)]
        (.fromBytes decimal-conversion msg schema logical-type))

      :else (.array msg))

    Schema$Type/MAP
    (zipmap (map name (keys msg))
            (map (partial ->clj (.getValueType schema))
                 (vals msg)))

    Schema$Type/RECORD
    ;; Transients are making this slower, I wonder why?
    (loop [fields (seq (.. msg getSchema getFields)) record {}]
      (if-let [f (first fields)]
        (let [n (.name f)
              v (->clj (.schema f) (.get msg n))
              k (-> n keyword)]
          (recur (rest fields)
                 (assoc record k v)))
        record))

    (throw (Exception. (format "Field `%s` is not in schema" schema)))))
