(ns kafkakit.component.consumer
  (:require
    [com.stuartsierra.component :as component]
    [franzy.clients.consumer.callbacks :as callbacks]
    [franzy.clients.consumer.client :as consumer]
    [franzy.clients.consumer.defaults :as cd]
    [franzy.clients.consumer.protocols :as franzy]
    [kafkakit.consumer.protocols :refer [handler]]
    [taoensso.timbre :refer [info warn]]))

(defn active-consumer-key
  "Gets the key of the currently active consumer."
  [config]
  (:active-consumer config))

(defn active-consumer-config
  "Gets the configuration of the currently active consumer."
  [config]
  (let [active-k (active-consumer-key config)]
    (-> config :consumers active-k)))

(defn consumer-group-base-name
  "Gets the base name for all Kafka consumer groups in this application."
  [config]
  (:consumer-group-base-name config))

(defn consumer-group-id [config]
  "Builds a consumer group id for the currently active consumer."
  (let [consumer-config (active-consumer-config config)
        k (active-consumer-key config)
        base (consumer-group-base-name config)
        kstr (if (keyword? k) (name k) k)]
    (or (:group.id consumer-config)
        (str base "." kstr))))

(defn active-consumer-component [component]
  "Finds the active consumer component in the system."
  (get component (active-consumer-key (:config component))))

(def rebalance-listener
  (callbacks/consumer-rebalance-listener
    (fn [topic-partitions]
      (info :msg "Kafka consumer topic partitions assigned."
            :event "kafka.consumer.topic-partitions.assigned"
            :topic-partitions topic-partitions))
    (fn [topic-partitions]
      (info :msg "Kafka consumer topic partitions revoked."
            :event "kafka.consumer.topic-partitions.revoked"
            :topic-partitions topic-partitions))))

(defn- make-consumer [component]
  (let [config                       (:config component)
        cc {:bootstrap.servers       (:bootstrap.servers config)
            :group.id                (consumer-group-id config)
            :enable.auto.commit      (:enable.auto.commit config true)
            :auto.offset.reset       (:auto.offset.reset config :earliest)
            :auto.commit.interval.ms (:auto.commit.interval.ms config 1000)
            :key.deserializer        (:key.deserializer config
                                       "io.confluent.kafka.serializers.KafkaAvroDeserializer")
            :value.deserializer      (:value.deserializer config
                                       "io.confluent.kafka.serializers.KafkaAvroDeserializer")
            :request.timeout.ms      (:request.timeout.ms config 305000)
            :schema.registry.url     (:schema.registry.url config)
            :session.timeout.ms      (:session.timeout.ms config 300000)}
        options (cd/make-default-consumer-options
                  (merge (:options component)
                         {:rebalance-listener-callback rebalance-listener}))
        consumer (consumer/make-consumer cc options)]
    (info :msg "Created kafka consumer group."
          :event "kafka.consumer.consumer-created"
          :group.id (:group.id cc))
    consumer))

(defprotocol Subscriber
  (subscribe!!
    [component topic handler]
    "Subscribes synchronously, polling forever while events come in.
    Params:
      component:  The component.
      topic:      String name for the message. Ex: \"brands\"")
  (unsubscribe!
    [component]
    "Unsubscribe from all partitions.
    Params:
      component:  The component.
      topic:      String name for the message. Ex: \"brands\""))

(defrecord Consumer [config]
  Subscriber
  (subscribe!! [this topics handler]
    (when (-> this :config :enabled?)
      (let [consumer (:consumer this)
            active-consumer (active-consumer-component this)]
        (franzy/subscribe-to-partitions! consumer topics)
        (info :msg (str "Kafka consumer subscribed to topics " topics)
              :event "kafka.consumer.subscribed"
              :topics topics)
        (reset! (:subscribed this) true)
        (while (not @(:shutdown this))
          (Thread/sleep 1000) ;; avoid CPU thrashing when no messages
          (let [consumer-record (franzy/poll! consumer)]
            (handler active-consumer topics consumer-record)))
        (unsubscribe! this))))

  (unsubscribe! [this]
    (when (-> this :config :enabled?)
      (let [consumer (:consumer this)
            active-consumer (active-consumer-component this)]
        (franzy/clear-subscriptions! consumer)
        (info :msg (str "Kafka consumer unsubscribed from all partitions.")
              :event "kafka.consumer.unsubscribed")
        (reset! (:subscribed this) false))))

  component/Lifecycle
  (start [this]
    (let [merged-options (if (:options config)
                           (cd/make-default-consumer-options (:options this))
                           (cd/make-default-consumer-options))
          this (assoc this :options merged-options)]
      (if (:enabled? config)
        ;; TODO: Insert options here.
        (let [consumer (make-consumer this)]
          (info :msg "Started Kafka consumer component."
                :event "kafka.consumer.started")
          (assoc this
                 :consumer consumer
                 :shutdown (atom false)
                 :subscribed (atom false)))
        (do
          (warn :msg (str "Kafka consumer component disabled! "
                          "Messages will not be consumed.")
                :event "kafka.consumer.disabled")
          this))))

  (stop [{:keys [consumer options] :as this}]
    (when consumer
      (reset! (:shutdown this) true)
      (while @(:subscribed this) (Thread/sleep 100))
      (.close consumer)
      (info :msg "Stopped Kafka consumer component."
            :event "kafka.consumer.stopped")
      (dissoc this :consumer :options :shutdown :subscribed))))

(defn consumer [config]
  (->Consumer config))
