(ns kafkakit.component.producer
  (:require
    [com.stuartsierra.component :as component]
    [franzy.clients.producer.client :as producer]
    [franzy.clients.producer.defaults :as pd]
    [franzy.clients.producer.protocols :refer [send-async!
                                               send-sync!]]
    [franzy.clients.producer.types :refer [make-producer-record]]
    [kafkakit.avro :as avro]
    [kafkakit.component.schema-registry :as registry]
    [taoensso.timbre :refer [info warn]]))

(defn make-producer-config [config]
  (let [pc {:bootstrap.servers (-> config :config :bootstrap.servers)
            :client.id         (-> config :config :client.id)
            :partitioner.class (-> config :config :partitioner.class)
            :key.serializer    (-> config :config :key.serializer)
            :value.serializer  (-> config :config :value.serializer)
            :schema.registry.url (-> config :schema-registry :schema-registry-url)}
        opt-ks [:buffer.memory :batch.size] ;; probably can do better than having to update hardcoded list of optional config keys
        opt-cs (map #(when-let [v (-> config :config %)] {% v}) opt-ks)]
    (apply merge (cons pc opt-cs))))

(defn make-producer [config]
  (let [pc (make-producer-config config)]
    (producer/make-producer pc (:options config))))

(defn create-producer-record [topic k v registry]
  (let [p nil
        kschema (registry/latest-schema registry (str topic "-key"))
        vschema (registry/latest-schema registry (str topic "-value"))
        kencode (avro/->java kschema k)
        vencode (avro/->java vschema v)]
    (make-producer-record topic p kencode vencode)))

(defprotocol Publisher
  (publish!!
    [component topic k v]
    "Publishes synchronously, blocking until the write is acknowledged.
    Params:
      component:  The component.
      topic:      Topic name for the event. Ex: \"audiences\"
      k:          Kafka message key.
      v:          Kafka message value.
    ")

  (publish-async!!
    [component topic k v]
    "Publishes asynchronously, non-blocking on write.
    Returns Future of RecordMetadata that can optionally be polled for (blocks if done).
    Params:
      component:  The component.
      topic:      Topic name for the event. Ex: \"audiences\"
      k:          Kafka message key.
      v:          Kafka message value.
    "))

(defn send!! [component topic k v async?]
  (when (-> component :config :enabled?)
    (let [record (create-producer-record topic k v (:schema-registry component))
          send-f (if async? send-async! send-sync!)]
      (send-f (:producer component) record))))

(defrecord Producer [config]
  component/Lifecycle
  ;; Create the KafkaProducer and open a connection to the broker.
  (start [this]
    (let [p-opts (if (:options config)
                   (pd/make-default-producer-options (:options this))
                   (pd/make-default-producer-options))
          p-conf (assoc this :options p-opts)]
      (if-let [enabled? (:enabled? config)]
        (let [producer (make-producer p-conf)]
          (info :msg "Started KafkaProducer component.")
          (assoc p-conf :producer producer))
        (do
          (warn :msg "KafkaProducer component disabled! Events will not be saved.")
          p-conf))))

  ;; Ensure the producer connection to the KafkaProducer broker is closed.
  (stop
    [{:keys [producer options] :as this}]
    (when producer (.close producer)
      (info :msg "Stopped KafkaProducer component.")
      (dissoc this :producer :options)))

  Publisher
  (publish!!
    [this topic k v]
    (send!! this topic k v false))

  (publish-async!!
    [this topic k v]
    (send!! this topic k v true)))

(defn producer [config]
  (->Producer config))
