(ns kafkakit.component.schema-registry
  (:require
    [clojure.core.memoize :as memo]
    [com.stuartsierra.component :as component]
    [kafkakit.schema-registry :as schema-registry]
    [taoensso.timbre :refer [info warn]]))

(defprotocol SchemaRegistryAvroFetcher
  (latest-schema [component subject]
    "Reads the latest Avro schema with the given subject name from the schema
    registry.

    Params:
      component: The component.
      subject: The Schema Registry topic for which the schema will be fetched."))

(defrecord SchemaRegistry
  [config]
  component/Lifecycle
  ;; Create the Kafka producer and open a connection to the broker.
  (start [this]
    (if-let [registry-client (schema-registry/cached-client config)]
      (do
        (info :msg "Started SchemaRegistry component."
              :schema-registry-url (:schema-registry-url config)
              :registry-cache-size (:registry-cache-size config))
        (assoc this :avro-schema-cache (atom {})
                    :registry-client registry-client
                    :schema-registry-url (:schema-registry-url config)))
      (warn :msg "Couldn't start SchemaRegistry component.")))

  ;; Ensure the producer connection to the Kafka broker is closed.
  (stop
    [this]
      (when (:registry-client this)
        (info :msg "Stopped SchemaRegistry component.")
        (dissoc this :registry-client :avro-schema-cache :schema-registry-url)))

  SchemaRegistryAvroFetcher
  (latest-schema [this schema-name]
    (let [fetch-f #(schema-registry/latest-schema (:schema-registry-url this) %)
          memo-f (memo/ttl fetch-f {} :ttl/threshold 300000)]
      (memo-f schema-name))))

(defn schema-registry [config]
  (->SchemaRegistry config))
