(ns plinth.db.sql
  (:require
    [plinth.schema :as s]
    [clojure.string :as string]))

(def SQLFragmentSchema
  { :sql s/Str
    :parameters (s/sequence-of s/Any) })

(defrecord SQLFragment [^String sql parameters])

(defn- quote-identifier [^String identifier]
  (->>
    (string/split identifier #"\.")
    (mapv #(format "\"%s\"" %))
    (string/join ".")))

(defn- to-fragment [value]
  (cond
    (instance? SQLFragment value)
                      value
    (instance? Boolean value)
                      (SQLFragment. (str value) [])
    (nil? value)      (SQLFragment. "NULL" [])
    (keyword? value)  (SQLFragment. (if (namespace value)
                                        (str  (quote-identifier (namespace value))
                                              "."
                                              (quote-identifier (name value)))
                                        (quote-identifier (name value)))
                                    [])
    (float? value)    (SQLFragment. (str value) [])
    (integer? value)  (SQLFragment. (str value) [])
    (string? value)   (SQLFragment. (str "'" (string/replace value "'" "''") "'") [])
    (or (set? value)
        (map? value)
        (sequential? value))
                      (if (empty? value)
                        (SQLFragment. "NULL" [])
                        (SQLFragment. "?" [value]))
    (instance? java.util.UUID value)
                      (SQLFragment. (str "'" value "'") [])))

(defn safe-format [fmt-str & values]
  (let [fragments (map to-fragment values)]
    (SQLFragment.
      (apply format fmt-str (map :sql fragments))
      (apply concat (map :parameters fragments)))))

(defn safe-infix [infix values]
  (let [fragments (map to-fragment values)]
    (SQLFragment.
      (string/join infix (map :sql fragments))
      (apply concat (map :parameters fragments)))))

(defn raw [sql & params]
  (SQLFragment. sql params))

(defn array [values]
  (if (not-empty values)
    (SQLFragment. (format "ARRAY[%s]" (string/join ", " (map (constantly "?") values))) values)
    (to-fragment nil)))

(defn ->sql [col]
  (->
    col
    (string/lower-case)
    (string/replace \- \_)))

(defn- sql-name-impl [col-name]
  (if (namespace col-name)
    (keyword (->sql (namespace col-name)) (->sql (name col-name)))
    (keyword (->sql (name col-name)))))

(def sql-name (memoize sql-name-impl))

(defn- ddl-col-for-type [schema col-name sql-type]
  (->>
    [(safe-format "%s %s" (sql-name col-name) (raw sql-type))
     (when (:required schema) (raw "NOT NULL"))
     (when (:unique schema) (raw "UNIQUE"))
     (when (= :relation (:type schema))
      (safe-format
        "REFERENCES %s (%s)"
        (-> schema :to :name sql-name)
        (-> schema :to :primary-key sql-name)))
     (when (:belongs-to? schema)
      (raw "ON DELETE CASCADE"))]
    (filter identity)
    (safe-infix " ")))

(defmulti ddl-col #'s/dispatch)
(defmethod ddl-col :serial [schema col-name] (ddl-col-for-type schema col-name "serial"))
(defmethod ddl-col :any [schema col-name] (ddl-col-for-type schema col-name "bytea"))
(defmethod ddl-col :string [schema col-name] (ddl-col-for-type schema col-name "text"))
(defmethod ddl-col :enum [schema col-name] (ddl-col-for-type schema col-name "text"))
(defmethod ddl-col :number [schema col-name] (ddl-col-for-type schema col-name "numeric"))
(defmethod ddl-col :integer [schema col-name] (ddl-col-for-type schema col-name "int"))
(defmethod ddl-col :uuid [schema col-name] (ddl-col-for-type schema col-name "uuid"))
(defmethod ddl-col :instant [schema col-name] (ddl-col-for-type schema col-name "timestamp with time zone"))
(defmethod ddl-col :boolean [schema col-name] (ddl-col-for-type schema col-name "boolean"))
(defmethod ddl-col :time [schema col-name] (ddl-col-for-type schema col-name "time"))
(defmethod ddl-col :date [schema col-name] (ddl-col-for-type schema col-name "date"))
(defmethod ddl-col :set [schema col-name] (ddl-col-for-type schema col-name "json"))
(defmethod ddl-col :sequence [schema col-name] (ddl-col-for-type schema col-name "json"))
(defmethod ddl-col :uri [schema col-name] (ddl-col-for-type schema col-name "text"))
(defmethod ddl-col :relation [schema col-name]
  (ddl-col-for-type schema col-name
    (case (get-in schema [:to :subschema (-> schema :to :primary-key) :type])
      :serial  "int"
      :integer "int"
      :string  "text"
      :uuid    "uuid"
      "unknown")))

(defn ddl-table [schema]
  (safe-format "CREATE TABLE IF NOT EXISTS %s (\n\t%s)"
    (sql-name (:name schema))
    (safe-infix ",\n\t"
      (mapv #(ddl-col (val %) (key %)) (:subschema schema)))))
