(ns plinth.core
  (:require
    [clojure.edn :as edn]
    [clojure.java.io :as io]
    [environ.core :as environ]
    [clojure.repl :refer :all]
    [clojure.tools.logging :as log]
    [clojure.tools.namespace.repl :as repl]
    [com.stuartsierra.component :as component]))

(repl/disable-reload!)

(defmacro defsystem
  "Convenience macro to build a system"
  [fname system-map]
  `(defn ~fname [] (component/system-map ~@system-map)))

(def system nil)
(def env environ/env)
(def ^:private env-files nil)
(def ^:private system-init nil)

(defn set-init! [init]
  (alter-var-root #'system-init (constantly init)))

(defn set-env-files! [files]
  (alter-var-root #'env-files (constantly files)))

(defn add-env-file! [file]
  (alter-var-root #'env-files conj file))

(defn load-env! [filename]
  (try
    (alter-var-root #'env merge (edn/read-string (slurp (io/file filename))))
    (catch Exception e
      (log/warn e "Could not load environment from file" filename))))

(defn- stop-system [s]
  (when s (component/stop s)))

(defn initialize []
  (alter-var-root #'env (constantly environ/env))

  (doseq [file env-files]
    (load-env! file))

  (if system-init
    (do (alter-var-root #'system #(do (stop-system %) (system-init))) :ok)
    (throw (Error. "No system initializer function found."))))

(defn start []
  (alter-var-root #'system component/start)
  :started)

(defn stop []
  (alter-var-root #'system stop-system)
  :stopped)

(defn go []
  (initialize)
  (start))

(defn clear []
  (alter-var-root #'system #(do (stop-system %) nil))
  :ok)

(defn reset []
  (clear)
  (repl/refresh :after 'plinth.core/go))

(defn refresh []
  (repl/refresh))

(defn setup [src-dirs system-var & env-files]
  (apply repl/set-refresh-dirs (seq src-dirs))
  (set-env-files! env-files)
  (set-init! system-var))

(defn -main
  "Start the system that is passed as an argument (as in the dev-run task)"
  [system-str & env-files]
  (let [system-var (resolve (symbol system-str))]
    (set-env-files! env-files)
    (set-init! system-var)
    (go)))

(defn wrap-tests [system-var & env-files]
  (fn [tests]
    (set-env-files! env-files)
    (set-init! system-var)
    (go)
    (tests)
    (stop)))
