(ns plinth.table.sql
  (:require
    [plinth.db :as db]
    [plinth.db.sql :as sql]
    [plinth.table :as table]
    [plinth.cache :as cache]
    [com.stuartsierra.component :as component]))

(defn- conditional-put [condition cache key value & [expires]]
  (when condition
    (cache/put cache key value expires))
  condition)

(defn- conditional-delete [condition cache key]
  (when condition
    (cache/delete cache key))
  condition)

(defrecord SQLTable [schema]
  component/Lifecycle
  (start [this]
    (db/transact! (:db this) identity (sql/ddl-table schema))
    (assoc this
      :pk (:primary-key schema)
      :sql-to-clj-keys (into {} (mapv #(vector (name (sql/sql-name %)) (name %)) (keys (:subschema schema))))))
  (stop [this]
    (dissoc this :db))

  table/Table
  (look-up [{:keys [db pk cache sql-to-clj-keys] :as table} pv]
    (cache/through cache [(:name schema) pv]
      (->
        (db/query db sql-to-clj-keys
          (sql/safe-format
            "SELECT %s FROM %s WHERE %s = %s LIMIT 1"
            (sql/safe-infix ", " (keys sql-to-clj-keys))
            (sql/sql-name (:name schema))
            (sql/sql-name pk)
            (sql/raw "?" pv)))
        (first))))
  (insert! [{:keys [db pk cache sql-to-clj-keys] :as table} value]
    (let [pv
          (->
            (db/transact! db sql-to-clj-keys
              (sql/safe-format
                "INSERT INTO %s (%s) VALUES (%s) RETURNING %s"
                (sql/sql-name (:name schema))
                (sql/safe-infix ", " (map sql/sql-name (keys value)))
                (sql/safe-infix ", " (map #(sql/raw "?" %) (vals value)))
                (sql/sql-name pk)))
            (ffirst)
            (get pk))]
      (conditional-put pv cache [(:name schema) pv] (assoc value pk pv))))
  (update! [{:keys [db pk cache schema sql-to-clj-keys] :as table} pv value]
    (->
      (db/transact! db sql-to-clj-keys
        (sql/safe-format
          "UPDATE %s SET %s WHERE %s = %s"
          (sql/sql-name (:name schema))
          (sql/safe-infix ", "
            (mapv #(sql/safe-format "%s = %s" (sql/sql-name (key %)) (sql/raw "?" (val %))) value))
          (sql/sql-name pk)
          (sql/raw "?" pv)))
      (first)
      (= 1)
      (conditional-delete cache [(:name schema) pv])))
  (delete! [{:keys [db pk cache schema sql-to-clj-keys] :as table} pv]
    (->
      (db/transact! db sql-to-clj-keys
        (sql/safe-format
          "DELETE FROM %s WHERE %s = %s"
          (sql/sql-name (:name schema))
          (sql/sql-name pk)
          (sql/raw "?" pv)))
      (first)
      (= 1)
      (conditional-delete cache [(:name schema) pv]))))
