(ns raid.tcp
  (:require [raid.envelop :as envelop]
            [raid.handlers :as handlers]
            [raid.logging :as logging]
            [raid.promises :as promises]
            [raid.recipes :as recipes]
            [raid.resources :refer [byte-buffer-allocate byte-buffer-get byte-buffer-rewind
                                    byte-buffer-wrap handler-worker queue-put
                                    queue-take start-worker thread-interrupt uuid4
                                    try-close]]
            raid.server
            [raid.socket :as socket])
  (:import [java.io DataInputStream IOException InterruptedIOException PipedInputStream PipedOutputStream]
           java.net.InetSocketAddress
           java.nio.ByteBuffer
           [java.nio.channels AsynchronousServerSocketChannel AsynchronousSocketChannel AsynchronousCloseException ClosedChannelException]
           [java.util.concurrent ExecutionException LinkedBlockingDeque]
           [raid.exceptions EncodeInvalid RemoteConnectionClosed])
  (:gen-class))

(def ^:const buf-size 64)

(defn- socket-read [^AsynchronousSocketChannel socket ^ByteBuffer buf]
  (deref (.read socket buf)))

(defn- read-buf [^AsynchronousSocketChannel socket buf]
  (byte-buffer-rewind buf)
  (let [position (socket-read socket buf)]
    (if (pos? position)
      (do
        (byte-buffer-rewind buf)
        (byte-buffer-get buf position))
      (throw (new RemoteConnectionClosed)))))

(defn- socket-write [^AsynchronousSocketChannel socket payload]
  (deref (.write socket (byte-buffer-wrap payload))))

(defn- feed-stream [^PipedOutputStream pipe-out reader]
  (.write pipe-out ^bytes (reader))
  (.flush pipe-out)
  (recur pipe-out reader))

(defn- parse-execution-exception
  ([status-state e t-group]
   (parse-execution-exception status-state e t-group (:via (Throwable->map e))))
  ([status-state e t-group exl]
   (if exl
     (if (= IOException (-> exl first :type))
       (do
         (reset! status-state :REMOTE-CLOSED))
       (recur status-state e t-group (next exl)))
     (throw e))))

(defn- read-socket [socket buf pipe-out status-state t-group]
  (try
    (feed-stream pipe-out #(read-buf socket buf))
    (catch ClosedChannelException e)
    (catch RemoteConnectionClosed e
      (reset! status-state :REMOTE-CLOSED))
    (catch ExecutionException e
      (parse-execution-exception status-state e t-group))
    (catch InterruptedException e)))

(defn- handler-dispatch [io get-payload t]
  (if (handlers/on-msg io (get-payload))
    (recur io get-payload t)))

(defn- write-socket [io socket queue]
  (socket-write socket (queue-take queue))
  (recur io socket queue))

(defn- recipes-get-by-action [r action]
  (get (recipes/recipes r) action))

(defn- socket-connect [^AsynchronousSocketChannel socket ^String host ^Integer port]
  (deref (.connect socket (new InetSocketAddress host port))))

(defn- socket-close [^AsynchronousSocketChannel socket status-state]
  (.close socket)
  (reset! status-state :CLOSED))

(defn- socket-get-remote-address [^AsynchronousSocketChannel socket]
  (.getRemoteAddress socket))

(defn- socket-get-address [^AsynchronousSocketChannel socket]
  (list (.getLocalAddress socket) (socket-get-remote-address socket)))

(defn- socket-open? [^AsynchronousSocketChannel socket]
  (.isOpen socket))

(defn- server-accept [^AsynchronousServerSocketChannel server]
  (deref (.accept server)))

(defn- new-socket [socket host port & [{:keys [encoding decoder encoder encoder-delimiter stream-input]
                                        :or {decoder envelop/utcode-unpack encoder envelop/utcode-pack}
                                        :as opt} initial-status recipes handlers loggers]]
  (promises/start-routines)
  (let [{:keys [encoder decoder]} (if-let [encoding (get envelop/encoding-list encoding)]
                                    encoding
                                    {:encoder encoder :decoder decoder})
        uid (uuid4)
        c-name (str "raid-tcp-" uid)
        status-state (atom (or initial-status :NEW))
        c (atom -1)
        loggers (or loggers (atom []))
        recipes (or recipes (atom {}))
        handlers (or handlers (atom {}))
        w-queue (new LinkedBlockingDeque)
        t-group (new ThreadGroup c-name)
        pipe-out (new PipedOutputStream)
        pipe-in (new PipedInputStream pipe-out)
        decoder-in (if stream-input
                     (stream-input pipe-in)
                     (if (= :msgpack encoding)
                       (new DataInputStream pipe-in)
                       pipe-in))
        io (reify
             Object
             (toString [_]
               (if (socket-open? socket)
                 (apply format "RAID TCP %s - OPEN %s <==> %s" uid (socket-get-address socket))
                 (format "RAID TCP %s - CLOSED" uid)))
             clojure.lang.Named
             (getName [_]
               c-name)
             raid.socket.ISocket
             (connect [this]
               (case @status-state
                 :NEW
                 (do
                   (socket-connect socket host port)
                   (->> (read-socket socket (byte-buffer-allocate buf-size) pipe-out status-state t-group)
                        (start-worker t-group (str "socket-reader:" (name this))))
                   (->> (handler-dispatch this #(handler-worker decoder decoder-in) pipe-in)
                        (start-worker t-group (str "handler-dispatch:" (name this))))
                   (->> (write-socket this socket w-queue)
                        (start-worker t-group (str "socket-writer:" (name this))))
                   (reset! status-state :OPEN))))
             (close [_]
               (try-close decoder-in)
               (socket-close socket status-state)
               (thread-interrupt t-group))
             (getAddress [_]
               (socket-get-remote-address socket))
             (status [_]
               @status-state)
             (statusAddListener [_ k cb]
               (add-watch status-state k cb))
             (statusRemoveListener [_ k]
               (remove-watch status-state k))
             (write [this msg]
               (.write this msg nil))
             (write [this msg delimiter]
               (if-not (= @status-state :OPEN)
                 (throw (new RemoteConnectionClosed (str "Connection with the server is" (name @status-state)))))
               (queue-put w-queue (encoder msg (or delimiter encoder-delimiter)))
               (logging/log-msg this :out msg))
             raid.handlers.IHandlers
             (addMsgHandler [_ handler]
               (let [id (swap! c inc)]
                 (swap! handlers assoc id handler)
                 id))
             (msgHandlers [_]
               (vals (deref handlers)))
             (rmMsgHandler [_ id]
               (swap! handlers dissoc id)
               true)
             raid.logging.ILogger
             (addLogger [_ handler]
               (swap! loggers conj handler))
             (getLoggers [_] @loggers)
             raid.recipes.IRecipes
             (addRecipe [_ action handler]
               (swap! recipes assoc action handler)
               true)
             (recipeByAction [this action]
               (recipes-get-by-action this action))
             (recipes [_]
               (deref recipes))
             (rmRecipe [_ action]
               (swap! recipes dissoc action)
               true))]
    (when (= @status-state :OPEN)
      (->> (read-socket socket (byte-buffer-allocate buf-size) pipe-out status-state t-group)
           (start-worker t-group (str "socket-reader:" (name io))))
      (->> (handler-dispatch io #(handler-worker decoder decoder-in) pipe-in)
           (start-worker t-group (str "handler-dispatch:" (name io))))
      (->> (write-socket io socket w-queue)
           (start-worker t-group (str "socket-writer:" (name io)))))
    (socket/status-add-listener io :default (fn [_ _ _ new-val]
                                              (case new-val
                                                :REMOTE-CLOSED
                                                (do
                                                  (socket/status-rm-listener io :default)
                                                  (socket/close io))
                                                nil)))
    io))

(defn- server-on-socket-status-changed [clients client key ref old-val new-val]
  (when (some #{new-val} [:CLOSED :REMOTE-CLOSED])
    (swap! clients #(filterv (complement #{client}) %))))

(defn- accept-connection [server clients recipes handlers loggers opt]
  (while true
    (let [io (server-accept server)]
      (if (and io (socket-open? io))
        (let [add (socket-get-remote-address io)
              client (new-socket io (socket/get-host add) (socket/get-port add) opt :OPEN recipes handlers loggers)]
          (socket/status-add-listener client :server (fn [& args] (apply server-on-socket-status-changed clients client args)))
          (swap! clients conj client))))))

(defn connect [host port & [{:keys [encoding decoder encoder encoder-delimiter stream-input]
                             :or {decoder envelop/utcode-unpack encoder envelop/utcode-pack}
                             :as opt}]]
  (if (and encoding (not (some #{encoding} (keys envelop/encoding-list))))
    (throw (new EncodeInvalid (str encoding))))
  (new-socket (AsynchronousSocketChannel/open) host port opt))

(defn open-server [^Integer port & [{:keys [encoding decoder encoder encoder-delimiter stream-input]
                                     :or {decoder envelop/utcode-unpack encoder envelop/utcode-pack}
                                     :as opt}]]
  (if (and encoding (not (some #{encoding} (keys envelop/encoding-list))))
    (throw (new EncodeInvalid (str encoding))))
  (promises/start-routines)
  (let [uid (uuid4)
        c-name (str "raid-server-tcp-" uid)
        server (AsynchronousServerSocketChannel/open)
        status-state (atom :NEW)
        clients (atom [])
        c (atom -1)
        loggers (atom [])
        recipes (atom {})
        handlers (atom {})
        t-group (new ThreadGroup c-name)]
    (reify
      Object
      (toString [_]
        (apply format "RAID SERVER TCP %s - WAITING" uid))
      clojure.lang.Named
      (getName [_]
        c-name)
      raid.server.IServer
      (bind [this]
        (.bind server (new InetSocketAddress port))
        (reset! status-state :WAITING)
        (->> (accept-connection server clients recipes handlers loggers opt)
             (start-worker t-group (str "socket-accept:" (name this)))))
      (clients [_]
        (deref clients))
      (clients-add-listener [_ k cb]
        (add-watch clients k cb))
      (clients-rm-listener [_ k]
        (remove-watch clients k))
      (close [_]
        (thread-interrupt t-group)
        (doseq [client clients]
          (socket/close client))
        (.close server)
        (reset! status-state :CLOSED))
      (status [_]
        @status-state)
      (status-add-listener [_ k cb]
        (add-watch status-state k cb))
      (status-rm-listener [_ k]
        (remove-watch status-state k))
      raid.handlers.IHandlers
      (addMsgHandler [_ handler]
        (let [id (swap! c inc)]
          (swap! handlers assoc id handler)
          id))
      (msgHandlers [_]
        (vals (deref handlers)))
      (rmMsgHandler [_ id]
        (swap! handlers dissoc id)
        true)
      raid.logging.ILogger
      (addLogger [_ handler]
        (swap! loggers conj handler))
      (getLoggers [_] @loggers)
      raid.recipes.IRecipes
      (addRecipe [_ action handler]
        (swap! recipes assoc action handler)
        true)
      (recipeByAction [this action]
        (recipes-get-by-action this action))
      (recipes [_]
        (deref recipes))
      (rmRecipe [_ action]
        (swap! recipes dissoc action)
        true))))
