# RAID Client

[![Clojars Project](https://img.shields.io/clojars/v/br.com.vikingmakt/raid.svg)](https://clojars.org/br.com.vikingmakt/raid)

## INSTALL

### Leiningen/Boot

```clojure
[br.com.vikingmakt/raid "1.5.4"]
```

### Clojure CLI/deps.edn

```clojure
br.com.vikingmakt/raid {:mvn/version "1.5.4"}
```

### Gradle

```gradle
compile 'br.com.vikingmakt:raid:1.5.4'
```

### Maven

```xml
<dependency>
  <groupId>br.com.vikingmakt</groupId>
  <artifactId>raid</artifactId>
  <version>1.5.4</version>
</dependency>
```

## HOW TO

### CONNECTING TO SERVER

```clojure
(require 'raid.core)

(def connection (raid.core/tcp-connect "localhost" 8888))
```

OR

```clojure
(def connection (raid.core/udp-connect "localhost" 8888 {:local-port 8890}))
```

### REQUESTING

```clojure
(let [resp @(raid.core/request connection "utos.es.api.utos.timestamp")]
  (println resp))
```

```sh
>>> {
	:body 1517507374
	:header {
		:code "UTOS-00000"
		:etag "f01fea9b4be6de12f0f05adb92fcd33e6ef2a9cba258c460c7d32c18df54"
    }
}
```

## INCLUDING ANOTHER MESSAGE FORMAT

The raid client/server support the formats `:utcode` and `:msgpack` (default is `:utcode`):

```clojure
(let [connection (raid/connect-tcp
                   hostname port
                   {:encoding :msgpack
                    :stream-input #(new DataInputStream %)})]
  ...
```

If you need anything else, you can add a function to encode and a function to decode the messages:

```clojure
(let [connection (raid/connect-tcp
                   hostname port
                   {:decoder unpack
                    :encoder pack
                    :stream-input #(new DataInputStream %)})]
  ...
```

The unpacked message must have the keys `:ok?`, telling if the payload was decoded, and `:msg`, which is the decoded message, and `:err`, with the exception, or throw a `IOException` to tell the decoder thread to stop.

```clojure
{:ok? true :msg {:header {:code "UTOS-00000" :etag "123456789"}}}
```

### JSON

An example of json pack and unpack functions:

```clojure
(ns example.core
  (:require [cheshire.core :as cheshire]
            [msgpack.core :as msgpack]
            ;; Let's import the functions to validate the message
            [raid.envelop :refer [pack-action pack-etag pack-sysdate str->bytes]])
  (:gen-class))

(defn json-pack [p & [delimiter]]
  (thread-envelop
    (-> p
        pack-action
        pack-etag
        pack-sysdate
        cheshire/generate-string
        (str delimiter)
        str->bytes)))

(defn json-unpack [^InputStream s]
  (thread-envelop
    (try
      {:ok? true :msg (cheshire/parse-stream s true)}
      (catch IOException e
        (throw e))
      (catch Exception e
        {:ok? false :err e}))))

```

# CHANGELOG

## [1.5.5]

* Fix TCP client when default params used
* Fix UDP write method
* Set threads for envelop to 10 (before as 2)

## [1.5.4]

* Send message with delimiter

## [1.5.3]

* TCP server

## [1.5.2]

* Clients listeners
* Handlers hierarchy
* Status listeners

## [1.5.1]

* TCP server

## [1.4.0]

* Adding UDP support

## [1.3.1]

* Streamming reading

## [1.0.0]

* Rewrite TCP client
* Writing multiple messages once
* Fix socket reader discarting messages if server writes too fast
* Remove UDP client (temporary)

## [0.3.8]

* Fix Broken Pipe exception when trying to write on a closed socket

## [0.3.7]

* Queue TCP socket write

## [0.3.6]

* Using augustus lib

## [0.3.5]

* Recipes

## [0.3.4]

* Bug fix - wrong function positioning

## [0.3.3]

* Wait TCP connection to be ready

## [0.3.2]

* Removing NO_DELAY option due Java socket bug

## [0.3.1]

* Bug fix - breakline by message

## [0.3.0]

* Changing TCP lib to java.nio.
* Rewrite thread pool call and thread pool as callback

## [0.2.3]

* Using thread pool to async operations instead of common threads

## [0.2.2]

* Logging handlers

## [0.2.1]

* Using defmulti to call shared functions
* Function "request" created

## [0.2.0]

* Connection object as record
* Separate handlers for messages and exceptions

## [0.1.1]

* Remove "connect" function from tcp/udp
* Base functions for common operations

## [0.0.3]

* Thread start function
* Using macros instead of lambdas

## [0.0.2]

* Some recipes and cancel socket listening

## [0.0.1]

* Socket creation and socket listening
