(ns raid.tcp
  (:require [raid.envelop :as envelop]
            [raid.handlers :as handlers]
            [raid.logging :as logging]
            [raid.promises :as promises]
            [raid.recipes :as recipes]
            [raid.resources :refer [uuid4]]
            raid.socket)
  (:import java.io.IOException
           java.net.InetSocketAddress
           java.nio.ByteBuffer
           [java.nio.channels AsynchronousSocketChannel AsynchronousCloseException]
           [java.util.concurrent ExecutionException LinkedBlockingDeque])
  (:gen-class))

(defn- queue-put [^LinkedBlockingDeque queue v]
  (.put queue v))

(defn- queue-take [^LinkedBlockingDeque queue]
  (.take queue))

(defn- byte-buffer-rewind [^ByteBuffer buf]
  (.rewind buf))

(defn- socket-read [^AsynchronousSocketChannel socket ^ByteBuffer buf]
  (deref (.read socket buf)))

(defn- read-buf [^AsynchronousSocketChannel socket ^ByteBuffer buf]
  (byte-buffer-rewind buf)
  (socket-read socket buf)
  (subvec (into [] (.array buf)) 0 (.position buf)))

(defn- socket-write [^AsynchronousSocketChannel socket ^bytes payload]
  (deref (.write socket (ByteBuffer/wrap payload))))

(defn- parse-buffer [io reader data msg callback]
  (if-let [c (first data)]
    (if (= c 10)
      (do
        (callback msg)
        (recur io reader (rest data) [] callback))
      (recur io reader (rest data) (conj msg c) callback))
    (recur io reader (reader) msg callback)))

(defn- read-socket [socket buf io decoder]
  (parse-buffer io #(read-buf socket buf) [] [] #(handlers/on-msg io decoder %)))

(defn- write-socket [io socket queue]
  (socket-write socket (queue-take queue))
  (recur io socket queue))

(defn- recipes-get-by-action [r action]
  (get (recipes/recipes r) action))

(defn- socket-connect [^AsynchronousSocketChannel socket ^String host ^Integer port]
  (deref (.connect socket (new InetSocketAddress host port))))

(defn- socket-close [^AsynchronousSocketChannel socket]
  (.close socket))

(defn- socket-get-remote-address [^AsynchronousSocketChannel socket]
  (.getRemoteAddress socket))

(defn- socket-get-address [^AsynchronousSocketChannel socket]
  (list (.getLocalAddress socket) (socket-get-remote-address socket)))

(defn- socket-open? [^AsynchronousSocketChannel socket]
  (.isOpen socket))

(defn- thread-interrupt [^ThreadGroup t]
  (.interrupt t))

(defn start-worker* [^ThreadGroup t-group t-name callable]
  (doto (new Thread t-group callable t-name)
    .start))

(defmacro start-worker [t-group t-name & body]
  `(start-worker*
    ~t-group
    ~t-name
    (fn []
      (try
        (do
          ~@body)
        (catch InterruptedException e#)
        (catch ExecutionException e#
          (let [e-map# (Throwable->map e#)]
            (case (:cause e-map#)
              IOException nil
              (throw e#))))))))

(defn connect [host port & [{:keys [decoder encoder] :or {decoder envelop/unpack encoder envelop/pack}}]]
  (promises/start-routines)
  (let [uid (uuid4)
        c-name (str "raid-tcp-" uid)
        socket (AsynchronousSocketChannel/open)
        c (atom -1)
        loggers (atom [])
        recipes (atom {})
        handlers (atom {})
        w-queue (new LinkedBlockingDeque)
        t-group (new ThreadGroup c-name)]
    (reify
      Object
      (toString [_]
        (if (socket-open? socket)
          (apply format "RAID TCP %s - OPEN %s <==> %s" uid (socket-get-address socket))
          (format "RAID TCP %s - CLOSED" uid)))
      clojure.lang.Named
      (getName [_]
        c-name)
      raid.socket.ISocket
      (connect [this]
        (socket-connect socket host port)
        (->> (read-socket socket (ByteBuffer/allocate 64) this decoder)
             (start-worker t-group (str "socket-reader:" (name this))))
        (->> (write-socket this socket w-queue)
             (start-worker t-group (str "socket-writer:" (name this)))))
      (close [_]
        (thread-interrupt t-group)
        (socket-close socket))
      (getAddress [_]
        (socket-get-remote-address socket))
      (write [this msg]
        (queue-put w-queue (encoder msg))
        (logging/log-msg this :out msg))
      raid.handlers.IHandlers
      (addMsgHandler [_ handler]
        (let [id (swap! c inc)]
          (swap! handlers assoc id handler)
          id))
      (msgHandlers [_]
        (vals (deref handlers)))
      (rmMsgHandler [_ id]
        (swap! handlers dissoc id)
        true)
      raid.logging.ILogger
      (addLogger [_ handler]
        (swap! loggers conj handler))
      (getLoggers [_] @loggers)
      raid.recipes.IRecipes
      (addRecipe [_ action handler]
        (swap! recipes assoc action handler)
        true)
      (recipeByAction [this action]
        (recipes-get-by-action this action))
      (recipes [_]
        (deref recipes))
      (rmRecipe [_ action]
        (swap! recipes dissoc action)
        true))))
