(ns raid.envelop
  (:require [augustus.async :as async]
            [raid.resources :refer [datetime-float-str gen-etag uuid4]]
            [utcode.core :as utcode])
  (:import [java.util.concurrent Executors ExecutorService ThreadFactory]
           raid.exceptions.ActionMissing)
  (:gen-class))

(defn- set-thread-name [^Thread t c]
  (.setName t (str "envelop-thread-" (swap! c inc))))

(defn- factory []
  (let [c (atom 0)]
    (proxy [ThreadFactory] []
      (newThread [^Runnable r]
        (doto (new Thread r)
          (set-thread-name c))))))

(defonce ^ExecutorService pool (Executors/newFixedThreadPool 2 (factory)))

(defn pack-action [p]
  (if-not (get-in p [:header :action])
    (throw (new ActionMissing))
    p))

(defn pack-etag [p]
  (if-not (get-in p [:header :etag])
    (update-in p [:header :etag] (fn [_] (gen-etag)))
    p))

(defn pack-sysdate [p]
  (if-not (contains? p :__sysdate__)
    (update-in p [:header :__sysdate__] (fn [_] (datetime-float-str)))
    p))

(defn bytes->str [ba]
  (let [sb (new StringBuilder)]
    (doseq [b ba]
      (.append sb (char b)))
    (str sb)))

(defn str->bytes [s]
  (byte-array (map int s)))

(defn pack [p]
  (deref
   (async/custom-submit
    pool
    (-> p
        pack-action
        pack-etag
        pack-sysdate
        utcode/encode
        (str \newline)
        str->bytes))))

(defn unpack [p]
  (deref
   (async/custom-submit
    pool
    (try
      {:ok? true :msg (utcode/decode (bytes->str p))}
      (catch Throwable e
        {:ok? false :err e})))))
