(ns raid.tcp
  (:require [raid.envelop :as envelop]
            [raid.handlers :as handlers]
            [raid.promises :as promises]
            raid.recipes
            [raid.resources :refer [uuid4]]
            raid.socket)
  (:import java.io.IOException
           java.net.InetSocketAddress
           java.nio.ByteBuffer
           [java.nio.channels AsynchronousSocketChannel AsynchronousCloseException]
           [java.util.concurrent ExecutionException LinkedBlockingDeque])
  (:gen-class))

(defn- queue-poll [^LinkedBlockingDeque queue]
  (.poll queue))

(defn- queue-put [^LinkedBlockingDeque queue v]
  (.put queue v))

(defn- queue-take [^LinkedBlockingDeque queue]
  (.take queue))

(defn- constr-str-recur* [^StringBuilder s values]
  (if (seq values)
    (recur (.append s (first values)) (rest values))
    (str s)))

(defn- constr-str [values]
  (constr-str-recur* (new StringBuilder) values))

(defn- byte-buffer-rewind [^ByteBuffer buf]
  (.rewind buf))

(defn- socket-read [^AsynchronousSocketChannel socket ^ByteBuffer buf]
  (deref (.read socket buf)))

(defn- read-buf [^AsynchronousSocketChannel socket ^ByteBuffer buf]
  (byte-buffer-rewind buf)
  (socket-read socket buf)
  (map char (subvec (into [] (.array buf)) 0 (.position buf))))

(defn- socket-write [^AsynchronousSocketChannel socket payload]
  (deref (.write socket (ByteBuffer/wrap (.getBytes ^String payload "UTF-8")))))

(defn- parse-buffer [reader data msg callback]
  (if-let [c (first data)]
    (if (= c \newline)
      (do
        (callback msg)
        (recur reader (rest data) [] callback))
      (recur reader (rest data) (conj msg c) callback))
    (recur reader (reader) msg callback)))

(defn- read-socket [socket buf io decoder]
  (parse-buffer #(read-buf socket buf) [] [] #(handlers/on-msg io decoder (constr-str %))))

(defn- batch-payload [queue payload]
  (if-let [head (queue-poll queue)]
    (recur queue (conj payload head \newline))
    payload))

(defn- write-socket [socket queue]
  (let [msg (constr-str (batch-payload queue [(queue-take queue) \newline]))]
    (socket-write socket msg))
  (recur socket queue))

(defn- recipes-get-by-action [^raid.recipes.IRecipes r action]
  (get (.recipes r) action))

(defn- socket-connect [^AsynchronousSocketChannel socket ^String host ^Integer port]
  @(.connect ^AsynchronousSocketChannel socket (new InetSocketAddress host port)))

(defn- socket-close [^AsynchronousSocketChannel socket]
  (.close socket))

(defn- socket-get-remote-address [^AsynchronousSocketChannel socket]
  (.getRemoteAddress socket))

(defn- socket-get-address [^AsynchronousSocketChannel socket]
  (list (.getLocalAddress socket) (socket-get-remote-address socket)))

(defn- socket-open? [^AsynchronousSocketChannel socket]
  (.isOpen socket))

(defn- thread-interrupt [^Thread t]
  (.interrupt t))

(defn- tcp-client [encoder]
  (let [uid (uuid4)
        name (str "raid-tcp-" uid)
        socket (AsynchronousSocketChannel/open)
        c (atom -1)
        recipes (atom {})
        handlers (atom {})
        w-queue (new LinkedBlockingDeque)
        io-threads (new ThreadGroup name)]
    [io-threads
     socket
     w-queue
     (reify
       Object
       (toString [_]
         (if (socket-open? socket)
           (apply format "RAID TCP %s - OPEN %s <==> %s" uid (socket-get-address socket))
           (format "RAID TCP %s - CLOSED" uid)))
       clojure.lang.Named
       (getName [_]
         name)
       raid.socket.ISocket
       (close [_]
         (thread-interrupt io-threads)
         (socket-close socket))
       (getAddress [_]
         (socket-get-remote-address socket))
       (write [_ msg]
         (queue-put w-queue (encoder msg)))
       raid.handlers.IHandlers
       (addMsgHandler [_ handler]
         (let [id (swap! c inc)]
           (swap! handlers assoc id handler)
           id))
       (msgHandlers [_]
         (vals (deref handlers)))
       (rmMsgHandler [_ id]
         (swap! handlers dissoc id)
         true)
       raid.recipes.IRecipes
       (addRecipe [_ action handler]
         (swap! recipes assoc action handler)
         true)
       (recipeByAction [this action]
         (recipes-get-by-action this action))
       (recipes [_]
         (deref recipes))
       (rmRecipe [_ action]
         (swap! recipes dissoc action)
         true))]))

(defn connect [host port & {:keys [decoder encoder] :or {decoder envelop/unpack encoder envelop/pack}}]
  (promises/start-routines)
  (let [[io-threads socket w-queue io] (tcp-client encoder)]
    (socket-connect socket host port)
    (.start
     (new Thread
          ^ThreadGroup io-threads
          (fn []
            (try
              (read-socket socket (ByteBuffer/allocate 64) io decoder)
              (catch InterruptedException e)
              (catch ExecutionException e
                (let [e (Throwable->map e)]
                  (case (:cause e)
                    IOException
                    nil)))))
          (str "socket-reader:" (name io))))
    (.start
     (new Thread
          ^ThreadGroup io-threads
          (fn []
            (try
              (write-socket socket w-queue)
              (catch InterruptedException e)
              (catch ExecutionException e
                (let [e (Throwable->map e)]
                  (case (:cause e)
                    IOException
                    nil)))))
          (str "socket-writer:" (name io))))
    io))
