(ns raid.tcp
  (:require [raid.envelop :as envelop]
            [raid.handlers :as handlers]
            [raid.promises :as promises]
            raid.recipes
            [raid.resources :refer [uuid4]]
            raid.socket)
  (:import java.io.IOException
           java.net.InetSocketAddress
           java.nio.ByteBuffer
           [java.nio.channels AsynchronousSocketChannel AsynchronousCloseException]
           [java.util.concurrent ExecutionException LinkedBlockingDeque])
  (:gen-class))

(defn- read-buf [^AsynchronousSocketChannel socket ^ByteBuffer buf]
  (.rewind buf)
  (deref (.read socket buf))
  (map char (subvec (into [] (.array buf)) 0 (.position buf))))

(defn- parse-buffer [reader data msg callback]
  (if-let [c (first data)]
    (if (= c \newline)
      (do
        (callback msg)
        (recur reader (rest data) [] callback))
      (recur reader (rest data) (conj msg c) callback))
    (recur reader (reader) msg callback)))

(defn- read-socket [socket buf io decoder]
  (parse-buffer #(read-buf socket buf) [] [] #(handlers/on-msg io decoder (apply str %))))

(defn- batch-payload [^LinkedBlockingDeque queue payload]
  (if-let [head (.poll queue)]
    (recur queue (conj payload head \newline))
    payload))

(defn- write-socket [^AsynchronousSocketChannel socket ^LinkedBlockingDeque queue]
  (let [msg (apply str (batch-payload queue [(.take queue) \newline]))]
    @(.write socket (ByteBuffer/wrap (.getBytes ^String msg))))
  (recur socket queue))

(defn- tcp-client [encoder]
  (let [uid (uuid4)
        name (str "raid-tcp-" uid)
        socket (AsynchronousSocketChannel/open)
        c (atom -1)
        recipes (atom {})
        handlers (atom {})
        w-queue (new LinkedBlockingDeque)
        io-threads (new ThreadGroup name)]
    [io-threads
     socket
     w-queue
     (reify
       Object
       (toString [_]
         (if (.isOpen socket)
           (format "RAID TCP %s - OPEN %s <==> %s" uid  (.getLocalAddress socket) (.getRemoteAddress socket))
           (format "RAID TCP %s - CLOSED" uid )))
       clojure.lang.Named
       (getName [_]
         name)
       raid.socket.ISocket
       (close [_]
         (.interrupt io-threads)
         (.close socket))
       (getAddress [_]
         (.getRemoteAddress socket))
       (write [_ msg]
         (.put w-queue (encoder msg)))
       raid.handlers.IHandlers
       (addMsgHandler [_ handler]
         (let [id (swap! c inc)]
           (swap! handlers assoc id handler)
           id))
       (msgHandlers [_]
         (vals (deref handlers)))
       (rmMsgHandler [_ id]
         (swap! handlers dissoc id)
         true)
       raid.recipes.IRecipes
       (addRecipe [_ action handler]
         (swap! recipes assoc action handler)
         true)
       (recipeByAction [this action]
         (get (.recipes this) action))
       (recipes [_]
         (deref recipes))
       (rmRecipe [_ action]
         (swap! recipes dissoc action)
         true))]))

(defn connect [^String host ^Integer port & {:keys [decoder encoder] :or {decoder envelop/unpack encoder envelop/pack}}]
  (promises/start-routines)
  (let [[io-threads socket w-queue io] (tcp-client encoder)]
    (deref (.connect ^AsynchronousSocketChannel socket (new InetSocketAddress host port)))
    (.start
     (new Thread
          ^ThreadGroup io-threads
          (fn []
            (try
              (read-socket socket (ByteBuffer/allocate 64) io decoder)
              (catch InterruptedException e)
              (catch ExecutionException e
                (let [e (Throwable->map e)]
                  (case (:cause e)
                    IOException
                    nil)))))
          (str "socket-reader:" (name io))))
    (.start
     (new Thread
          ^ThreadGroup io-threads
          (fn []
            (try
              (write-socket socket w-queue)
              (catch InterruptedException e)
              (catch ExecutionException e
                (let [e (Throwable->map e)]
                  (case (:cause e)
                    IOException
                    nil)))))
          (str "socket-writer:" (name io))))
    io))
