(ns raid.promises
  (:refer-clojure :exclude [resolve promise])
  (:require [augustus.future :as afuture])
  (:import java.util.Calendar
           [java.util.concurrent Executors TimeoutException TimeUnit])
  (:gen-class))

(def default-timeout 5000)

(def ^:private promises (atom {}))

(defprotocol IPromise
  (cancel [this])
  (date [this])
  (isExpired [this] [this now])
  (resolve [this result]))

(defn promise-expired? [^raid.promises.IPromise prom & [now]]
  (if now
    (.isExpired prom now)
    (.isExpired prom)))

(defn promise-resolve [^raid.promises.IPromise prom result]
  (.resolve prom result))

(defn promise [etag fut & {:keys [timeout]}]
  (let [d (Calendar/getInstance)
        e (.clone d)]
    (.add ^Calendar e Calendar/MILLISECOND (or timeout default-timeout))
    (reify IPromise
      (cancel [_]
        (swap! promises dissoc etag)
        (afuture/cancel fut))
      (date [_]
        (.getTime d))
      (isExpired [this]
        (.isExpired this (Calendar/getInstance)))
      (isExpired [_ now]
        (.before ^Calendar e now))
      (resolve [_ result]
        (swap! promises dissoc etag)
        (try
          (do
            (fut result)
            true)
          (catch Throwable e
            false))))))

(defn add-promise [etag fut & {:keys [timeout]}]
  (let [prom (promise etag fut :timeout timeout)]
    (swap! promises assoc etag prom)
    prom))

(defn get-promise [etag]
  (get @promises etag))

(def ^:private routine-running (atom false))

(defn- check-timedout []
  (let [now (Calendar/getInstance)]
    (loop [promises (vals @promises)]
      (if (seq promises)
        (let [p (first promises)]
          (if (promise-expired? p now)
            (promise-resolve p (new TimeoutException)))
          (recur (rest promises)))))))

(defn start-routines []
  (when-not @routine-running
    (reset! routine-running true)
    (doto (Executors/newScheduledThreadPool 1)
      (.scheduleWithFixedDelay check-timedout 0 1 TimeUnit/SECONDS))))
