(ns raid.tcp
  (:require [raid.async :as async]
            [raid.base :as base])
  (:import java.net.InetSocketAddress
           java.nio.ByteBuffer
           [java.nio.channels AsynchronousSocketChannel AsynchronousCloseException CompletionHandler])
  (:gen-class))

(defn create-socket [host port]
  (doto (AsynchronousSocketChannel/open)
    (.connect (new InetSocketAddress host port))))

(defn send-msg [socket msg]
  (.write socket (ByteBuffer/wrap (.getBytes msg))))

(defn push-msg [io msg]
  (base/push-msg io msg #(send-msg (:socket io) %)))

(defn- read-socket [io buf handler]
  (.read (:socket io) buf handler handler))

(defn- on-excep [io ex]
  (base/call-handlers #(async/add-callback % io ex) @(:ex-handlers io)))

(defn- on-read [io data]
  (base/call-handlers #(async/add-callback base/on-msg io data %) @(:msg-handlers io)))

(defn- prepare-read [io buf]
  (read-socket io buf (proxy [CompletionHandler] []
                        (completed [result handler]
                          (when-not (= result -1)
                            (async/add-callback on-read io (new String (.array buf)))
                            (.clear buf)
                            (async/add-callback read-socket io buf handler)))
                        (failed [ex handler]
                          (if-not (instance? AsynchronousCloseException ex)
                            (async/add-callback on-excep io ex))))))

(defn create-buffer [size]
  (ByteBuffer/allocate size))

(defn start-read [io]
  (when-not (deref (:run io))
    (reset! (:run io) true)
    (prepare-read io (create-buffer @(:buf-size io)))))
